<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Url;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\gaya_popup\Entity\GayaPopup;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Gaya Popup.
 */
class GayaPopupBaseController extends ControllerBase {

  /**
   * Constructs the SchemaListenerController object.
   */
  public function __construct(protected Request $request, protected LoggerInterface $logger, protected AdminContext $adminContext, protected CacheBackendInterface $cache, protected TimeInterface $time, protected readonly RendererInterface $renderer) {
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('logger.factory')->get('popup_entity'),
      $container->get('router.admin_context'),
      $container->get('cache.default'),
      $container->get('datetime.time'),
      $container->get('renderer'),
    );
  }

  /**
   * Enable a gaya Popup.
   */
  public function enableMessage(GayaPopup $popupmessage): RedirectResponse|array {
    if (!$this->currentUser()
      ->hasPermission('edit popup messages entity entities')) {
      return new RedirectResponse(Url::fromRoute('<front>')->toString());
    }

    try {
      $popupmessage->set('status', '1');
      $popupmessage->save();
    }
    catch (\Exception $e) {

      $this->logger->error($e->getMessage());
    }

    $response = new RedirectResponse(Url::fromRoute('gaya_popup.popup_entity_settings')
      ->toString());
    $response->send();
    return [];
  }

  /**
   * Disable a gaya popup.
   */
  public function disableMessage(GayaPopup $popupmessage): RedirectResponse|array {
    if (!$this->currentUser()
      ->hasPermission('edit popup messages entity entities')) {
      return new RedirectResponse(Url::fromRoute('<front>')->toString());
    }

    try {
      $popupmessage->set('status', '0');
      $popupmessage->save();
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage());
    }

    $response = new RedirectResponse(Url::fromRoute('gaya_popup.popup_entity_settings')
      ->toString());
    $response->send();
    return [];
  }

  /**
   * Get popup message.
   */
  public function getPopupMessage(): JsonResponse|array {
    $data = '';
    $langcode = $this->languageManager()->getCurrentLanguage()->getId();
    if (!$this->adminContext->isAdminRoute()) {
      $state = $this->state();
      $popup_disabled = $state->get('gaya_popup.settings.popup_disabled', 0);
      if ($popup_disabled) {
        return new JsonResponse("");
      }

      // Get zone display.
      $zone = $this->request->query->get('zone', '');

      // Search all popup.
      $cache = $this->cache()->get('gaya_popup');
      $cache_data = $cache && $cache->valid ? $cache->data : [];
      if (!isset($cache_data[$langcode])) {
        $cache_data[$langcode] = $results = $this->getPopups($langcode);
        $this->cache->set('gaya_popup', $cache_data);
      }
      else {
        $results = $cache_data[$langcode];
      }

      // Filter popups.
      if ($results) {
        $popup_entity = $this->filterScheduled($results, $zone);

        // Check zone and display zone.
        if ($popup_entity) {
          // Display popup.
          $data = $this->renderEntity($popup_entity);
        }
      }
    }
    return new JsonResponse($data);
  }

  /**
   * Request to get all enable popup.
   */
  protected function getPopups(string $langcode): array {
    $popupStorage = $this->entityTypeManager()
      ->getStorage('gaya_popup_entity');
    $results = $popupStorage
      ->getQuery()
      ->accessCheck()
      ->sort('weight')
      ->condition('status', 1, '=')
      ->condition('langcode', $langcode)
      ->execute();
    return $popupStorage->loadMultiple($results);
  }

  /**
   * Get entity render.
   */
  protected function renderEntity($popup_entity) {
    $entity = $this->entityTypeManager()
      ->getViewBuilder('gaya_popup_entity')
      ->view($popup_entity);
    return $this->renderer->render($entity);
  }

  /**
   * Filter scheduled.
   */
  protected function filterScheduled(array $results, string $zone) {
    if (!empty($results)) {
      $current_time = $this->time->getRequestTime();

      $date = $this->getCurrentDate();
      foreach ($results as $popup_entity) {
        if ($popup_entity->isVisible($zone)) {
          $date_on = $popup_entity->get('field_date_on')->date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
          $date_off = $popup_entity->get('field_date_off')->date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
          if ($date >= $date_on && $date <= $date_off) {
            if (!$popup_entity->hasField('field_scheduled') || $popup_entity->field_scheduled->value != '1') {
              return $popup_entity;
            }
            else {
              if (!$popup_entity->hasField('field_scheduled_begin') || !$popup_entity->hasField('field_scheduled_end')) {
                continue;
              }
              $begin = empty($popup_entity->field_scheduled_begin->value) ? 0 : $popup_entity->field_scheduled_begin->value;
              $end = empty($popup_entity->field_scheduled_end->value) ? 0 : $popup_entity->field_scheduled_end->value;
              if ($begin < $end) {
                if ((strtotime((string) $begin) <= $current_time) && (strtotime((string) $end) >= $current_time)) {
                  return $popup_entity;
                }
              }
              else {
                if ((strtotime((string) $begin) >= $current_time) || (strtotime((string) $end) <= $current_time)) {
                  return $popup_entity;
                }
              }
            }
          }
        }
      }
    }
    return NULL;
  }

  /**
   * Get current date.
   *
   * @throws \Exception
   */
  protected function getCurrentDate(): string {
    $timezone = date_default_timezone_get();
    $date = new \DateTime('today', new \DateTimezone($timezone));
    $date = DrupalDateTime::createFromDateTime($date);
    return $date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
  }

}
