<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Services;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\gaya_popup\Entity\GayaPopupInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for Gaya Popup.
 */
class GayaPopupManager {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LanguageManagerInterface $languageManager,
    protected ConfigFactoryInterface $config,
    protected RequestStack $requestStack,
    protected AdminContext $adminContext,
    protected CacheBackendInterface $cache,
    protected TimeInterface $time,
    protected readonly RendererInterface $renderer,
  ) {}

  /**
   * Get popup message.
   */
  public function getPopupMessage(): string|array|Markup {
    $data = '';
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    if (!$this->adminContext->isAdminRoute()) {
      $popup_disabled = $this->config->get('gaya_popup.settings')->get('popup_disabled');
      if ($popup_disabled) {
        return '';
      }

      // Get zone display.
      $zone = $this->requestStack->getCurrentRequest()->getPathInfo();

      // Search all popup.
      $cache = $this->cache->get('gaya_popup');
      $cache_data = $cache && $cache->valid ? $cache->data : [];
      if (!isset($cache_data[$langcode])) {
        $cache_data[$langcode] = $results = $this->getPopups($langcode);
        $this->cache->set('gaya_popup', $cache_data, Cache::PERMANENT, ['gaya_popup']);
      }
      else {
        $results = $cache_data[$langcode];
      }

      // Filter popups.
      if ($results) {
        $popups = $this->filterZone($results, $zone);
        // Check zone and display zone.
        if ($popups) {
          // Display popup.
          $data = $this->getDatas($popups);
        }
      }
    }
    return $data;
  }

  /**
   * Request to get all enable popup.
   */
  protected function getPopups(string $langcode): array {
    $popupStorage = $this->entityTypeManager
      ->getStorage('gaya_popup_entity');
    $query = $popupStorage->getQuery()->accessCheck();
    $query->sort('weight');
    $query->condition('status', 1, '=');
    $query->condition('langcode', $langcode);
    $or = $query->orConditionGroup();
    $or->condition('field_date_off', $this->getCurrentDate(), '>');
    $or->notExists('field_date_off');
    $query->condition($or);
    $results = $query->execute();
    return $popupStorage->loadMultiple($results);
  }

  /**
   * Get datas of all popups.
   */
  protected function getDatas(array $popups) {
    $datas = [];
    foreach ($popups as $popup) {
      $datas[] = $this->getData($popup, $this->renderEntity($popup));
    }
    return $datas;
  }

  /**
   * Get entity render.
   */
  protected function renderEntity(GayaPopupInterface $popup) {
    $entity = $this->entityTypeManager
      ->getViewBuilder('gaya_popup_entity')
      ->view($popup);
    return $this->renderer->renderInIsolation($entity);
  }

  /**
   * Array of data for a popup.
   */
  protected function getData(GayaPopupInterface $popup, MarkupInterface $render): array {
    $scheduled = !$popup->hasField('field_scheduled') || $popup->get('field_scheduled')->getString();
    return [
      'html' => $render,
      'dateStart' => $popup->hasField('field_date_on') && $popup->get('field_date_on')->getString() ? $popup->get('field_date_on')->date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT) : '',
      'dateEnd' => $popup->hasField('field_date_off')  && $popup->get('field_date_off')->getString() ? $popup->get('field_date_off')->date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT) : '',
      'hourStart' => $scheduled && $popup->hasField('field_scheduled_begin') ? $popup->get('field_scheduled_begin')->getString() : '',
      'hourEnd' => $scheduled && $popup->hasField('field_scheduled_end') ? $popup->get('field_scheduled_end')->getString() : '',
      'closed' => $popup->hasField('field_closed') ? $popup->get('field_closed')->getString() : TRUE,
      'frequency' => $popup->hasField('field_frequency') ? $popup->get('field_frequency')->getString() : 'once',
      'id' => $popup->id(),
    ];
  }

  /**
   * Filter zone.
   */
  protected function filterZone(array $results, string $zone): array {
    $popups = [];
    if (!empty($results)) {
      foreach ($results as $popup_entity) {
        if ($popup_entity->isVisible($zone)) {
          $popups[] = $popup_entity;
        }
      }
    }
    return $popups;
  }

  /**
   * Get current date.
   *
   * @throws \Exception
   */
  protected function getCurrentDate(): string {
    $timezone = date_default_timezone_get();
    $date = new \DateTime('today', new \DateTimezone($timezone));
    $date = DrupalDateTime::createFromDateTime($date);
    return $date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
  }

}
