<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\gaya_popup\Entity\GayaPopupInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for deleting a Popup Messages Entity revision.
 *
 * @ingroup gaya_popup
 */
class GayaPopupRevisionDeleteForm extends ConfirmFormBase {


  /**
   * The Popup Messages Entity revision.
   *
   * @var \Drupal\gaya_popup\Entity\GayaPopupInterface
   */
  protected GayaPopupInterface $revision;

  /**
   * Form Revision Delete for entity Popup.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $gayaPopupStorage
   *   The entity storage.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   */
  public function __construct(
    protected EntityStorageInterface $gayaPopupStorage,
    protected Connection $connection,
    private readonly DateFormatterInterface $dateFormatter,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container) {
    $entity_manager = $container->get('entity_type.manager');
    return new static(
      $entity_manager->getStorage('gaya_popup_entity'),
      $container->get('database'),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getFormId(): string {
    return 'gaya_popup_entity_revision_delete_confirm';
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getQuestion(): TranslatableMarkup {
    return t('Are you sure you want to delete the revision from %revision-date?', ['%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime())]);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getCancelUrl(): Url {
    return new Url('entity.gaya_popup_entity.version_history', ['gaya_popup_entity' => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getConfirmText(): TranslatableMarkup {
    return t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function buildForm(array $form, FormStateInterface $form_state, $gaya_popup_entity_revision = NULL): array {
    $this->revision = $this->gayaPopupStorage->loadRevision($gaya_popup_entity_revision);
    $form = parent::buildForm($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->gayaPopupStorage->deleteRevision($this->revision->getRevisionId());

    $this->logger('content')->notice('Popup Messages Entity: deleted %title revision %revision.',
      [
        '%title' => $this->revision->label(),
        '%revision' => $this->revision->getRevisionId(),
      ]
    );
    $this->messenger()->addStatus(t('Revision from %revision-date of Popup Messages Entity %title has been deleted.',
      [
        '%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime()),
        '%title' => $this->revision->label(),
      ])
    );
    $form_state->setRedirect(
      'entity.gaya_popup_entity.canonical',
       ['gaya_popup_entity' => $this->revision->id()]
    );
    if ($this->connection->query('SELECT COUNT(DISTINCT vid) FROM {gaya_popup_entity_field_revision} WHERE id = :id', [':id' => $this->revision->id()])->fetchField() > 1) {
      $form_state->setRedirect(
        'entity.gaya_popup_entity.version_history',
         ['gaya_popup_entity' => $this->revision->id()]
      );
    }
  }

}
