<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Defines the Popup Messages Entity entity.
 *
 * @ingroup gaya_popup
 *
 * @ContentEntityType(
 *   id = "gaya_popup_entity",
 *   label = @Translation("Popup"),
 *   handlers = {
 *     "storage" = "Drupal\gaya_popup\GayaPopupStorage",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\gaya_popup\GayaPopupListBuilder",
 *     "views_data" = "Drupal\gaya_popup\Entity\GayaPopupViewsData",
 *     "translation" = "Drupal\gaya_popup\GayaPopupTranslationHandler",
 *
 *     "form" = {
 *       "default" = "Drupal\gaya_popup\Form\GayaPopupForm",
 *       "add" = "Drupal\gaya_popup\Form\GayaPopupForm",
 *       "edit" = "Drupal\gaya_popup\Form\GayaPopupForm",
 *       "delete" = "Drupal\gaya_popup\Form\GayaPopupDeleteForm",
 *     },
 *     "access" = "Drupal\gaya_popup\GayaPopupAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\gaya_popup\GayaPopupHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "gaya_popup_entity",
 *   data_table = "gaya_popup_entity_field_data",
 *   revision_table = "gaya_popup_entity_revision",
 *   revision_data_table = "gaya_popup_entity_field_revision",
 *   translatable = TRUE,
 *   admin_permission = "administer popup messages entity entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "vid",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode",
 *     "status" = "status",
 *     "weight" = "weight",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log"
 *   },
 *   links = {
 *     "canonical" = "/admin/content/popup/{gaya_popup_entity}",
 *     "add-form" = "/admin/content/popup/add",
 *     "edit-form" = "/admin/content/popup/{gaya_popup_entity}/edit",
 *     "delete-form" = "/admin/content/popup/{gaya_popup_entity}/delete",
 *     "version-history" = "/admin/content/popup/{gaya_popup_entity}/revisions",
 *     "revision" = "/admin/content/popup/{gaya_popup_entity}/revisions/{gaya_popup_entity_revision}/view",
 *     "revision_revert" = "/admin/content/popup/{gaya_popup_entity}/revisions/{gaya_popup_entity_revision}/revert",
 *     "revision_delete" = "/admin/content/popup/{gaya_popup_entity}/revisions/{gaya_popup_entity_revision}/delete",
 *     "translation_revert" = "/admin/content/popup/{gaya_popup_entity}/revisions/{gaya_popup_entity_revision}/revert/{langcode}",
 *     "collection" = "/admin/content/popups",
 *   },
 *   field_ui_base_route = "gaya_popup.settings"
 * )
 */
class GayaPopup extends RevisionableContentEntityBase implements GayaPopupInterface {

  use EntityChangedTrait;

  /**
   * An alias manager to find the alias for the current system path.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected AliasManagerInterface $aliasManager;

  /**
   * The path matcher.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected PathMatcherInterface $pathMatcher;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The current path.
   *
   * @var \Drupal\Core\Path\CurrentPathStack
   */
  protected CurrentPathStack $currentPath;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The default front page.
   *
   * @var string
   */
  protected string $frontPage;

  public function __construct(array $values, $entity_type, $bundle = FALSE, $translations = []) {
    parent::__construct($values, $entity_type, $bundle, $translations);
    $this->aliasManager = \Drupal::service('path_alias.manager');
    $this->pathMatcher = \Drupal::service('path.matcher');
    $this->requestStack = \Drupal::service('request_stack');
    $this->currentPath = \Drupal::service('path.current');
    $this->configFactory = \Drupal::service('config.factory');
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function preCreate(EntityStorageInterface $storage, array &$values) {
    parent::preCreate($storage, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function urlRouteParameters($rel): array {
    $uri_route_parameters = parent::urlRouteParameters($rel);

    if ($rel === 'revision_revert' && $this instanceof RevisionableInterface) {
      $uri_route_parameters[$this->getEntityTypeId() . '_revision'] = $this->getRevisionId();
    }
    elseif ($rel === 'revision_delete' && $this instanceof RevisionableInterface) {
      $uri_route_parameters[$this->getEntityTypeId() . '_revision'] = $this->getRevisionId();
    }

    return $uri_route_parameters;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    foreach (array_keys($this->getTranslationLanguages()) as $langcode) {
      $translation = $this->getTranslation($langcode);

      // If no owner has been set explicitly, make the anonymous user the owner.
      if (!$translation->getOwner()) {
        $translation->setOwnerId(0);
      }
    }

    // If no revision author has been set explicitly, make the gaya_popup_entity
    // owner the revision author.
    if (!$this->getRevisionUser()) {
      $this->setRevisionUserId($this->getOwnerId());
    }
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getName(): string {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setName(string $name): GayaPopupInterface {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getCreatedTime(): int {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setCreatedTime(int $timestamp): GayaPopupInterface {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getOwner(): UserInterface {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getOwnerId(): ?string {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setOwnerId($uid): GayaPopupInterface {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setOwner(UserInterface $account): GayaPopupInterface {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function isPublished() {
    return (bool) $this->getEntityKey('status');
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setPublished(bool $published): GayaPopupInterface {
    $this->set('status', (bool) $published);
    return $this;
  }

  /**
   * Return visibility of the popup.
   */
  public function isVisible(string $path): bool {
    if ($this->hasField('field_area') && $this->get('field_area')->getFieldDefinition()->getType() === 'string_long') {
      $pages = $this->get('field_area')->getString();
      if (!$pages) {
        return TRUE;
      }
      if (!$path) {
        return FALSE;
      }
      $frontpage = $this->getFrontPagePath();
      if ($frontpage === $path) {
        $match = $this->pathMatcher->matchPath('/', trim($pages));
        if ($match) {
          return TRUE;
        }
      }
      $path_alias = mb_strtolower($this->aliasManager->getAliasByPath($path));
      return $this->pathMatcher->matchPath($path_alias, trim($pages)) || (($path != $path_alias) && $this->pathMatcher->matchPath($path, trim($pages)));
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Popup Messages Entity entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Popup for administration.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Enabled'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 99,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    $fields['revision_translation_affected'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Revision translation affected'))
      ->setDescription(t('Indicates if the last edit of a translation belongs to current revision.'))
      ->setReadOnly(TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE);
    $fields['weight'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Weight'))
      ->setRequired(TRUE)
      ->setDescription(t('Priority weight for alert message'))
      ->setDefaultValue(0)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);
    return $fields;
  }

  /**
   * Gets the current front page path.
   *
   * @return string
   *   The front page path.
   */
  protected function getFrontPagePath() {
    // Lazy-load front page config.
    if (!isset($this->frontPage)) {
      $this->frontPage = $this->configFactory->get('system.site')
        ->get('page.front');
    }
    return $this->frontPage;
  }

}
