<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Entity\Controller;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a list controller for gaya_popup_entity entity.
 *
 * @ingroup gaya_popup_entity
 */
class MessagesListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('url_generator'),
      $container->get('date.formatter')
    );
  }

  /**
   * Constructs a new ContactListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $urlGenerator
   *   The url generator.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, protected UrlGeneratorInterface $urlGenerator, protected DateFormatterInterface $dateFormatter) {
    parent::__construct($entity_type, $storage);
  }

  /**
   * {@inheritdoc}
   *
   * We override ::render() so that we can add our own content above the table.
   * parent::render() is where EntityListBuilder creates the table using our
   * buildHeader() and buildRow() implementations.
   */
  #[\Override]
  public function render() {
    $build['description'] = [
      '#markup' => $this->t('<a href="@adminlink">Draggable view for alerts entities</a>', [
        '@adminlink' => $this->urlGenerator->generateFromRoute('gaya_popup.popup_entity_settings'),
      ]),
    ];

    $build['table'] = parent::render();
    return $build;
  }

  /**
   * {@inheritdoc}
   *
   * Building the header and content lines for the Popup Entities list.
   *
   * Calling the parent::buildHeader() adds a column for the possible actions
   * and inserts the 'edit' and 'delete' links as defined for the entity type.
   */
  #[\Override]
  public function buildHeader() {
    $header['id'] = $this->t('ID');
    $header['title'] = $this->t('Title');
    $header['body'] = $this->t('Message');
    $header['enabled'] = $this->t('Enabled');
    $header['field_frequency'] = $this->t('Frequency');
    $header['field_closed'] = $this->t('Do not show if closed');
    $header['field_date_on'] = $this->t('Date begin');
    $header['field_date_off'] = $this->t('Date end');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function buildRow(EntityInterface $entity): array {
    /**
     * @var Drupal\gaya_popup\Entity\GayaPopup $entity
     */
    $datetime_on = "";
    if (!empty($entity->field_date_on->value)) {
      $datetime_on = strtotime((string) $entity->field_date_on->value);
      $datetime_on = $this->dateFormatter->format($datetime_on, 'custom', 'Y-m-d');
    }
    $datetime_off = "";
    if (!empty($entity->field_date_off->value)) {
      $datetime_off = strtotime((string) $entity->field_date_off->value);
      $datetime_off = $this->dateFormatter->format($datetime_off, 'custom', 'Y-m-d');
    }

    $row['id'] = $entity->id();
    $row['title'] = $entity->toLink()->toString();
    $row['body'] = $entity->body->value;
    $row['enabled'] = ($entity->enabled->value) ? $this->t("Yes") : $this->t("No");
    $row['field_frequency'] = ($entity->field_frequency->value == 'repeat') ? $this->t("Every Time") : $this->t("Once in session");
    $row['field_closed'] = ($entity->field_closed->value) ? $this->t("Yes") : $this->t("No");
    $row['field_date_on'] = $datetime_on;
    $row['field_date_off'] = $datetime_off;
    return $row + parent::buildRow($entity);
  }

}
