<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Controller;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\gaya_popup\Entity\GayaPopupInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class GayaPopupController.
 *
 *  Returns responses for Popup Messages Entity routes.
 */
class GayaPopupController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * Constructs a new GayaPopupController.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   */
  public function __construct(
    protected RendererInterface $renderer,
    protected DateFormatterInterface $dateFormatter,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('date.formatter')
    );
  }

  /**
   * Displays a Popup Messages Entity  revision.
   *
   * @param int $gaya_popup_entity_revision
   *   The Popup Messages Entity  revision ID.
   *
   * @return array
   *   An array suitable for drupal_render().
   */
  public function revisionShow($gaya_popup_entity_revision) {
    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $gaya_popup_entity */
    $gaya_popup_entity = $this->entityTypeManager()->getStorage('gaya_popup_entity');
    $gaya_popup_entity->loadRevision($gaya_popup_entity_revision);
    $view_builder = $this->entityTypeManager()->getViewBuilder('gaya_popup_entity');

    return $view_builder->view($gaya_popup_entity);
  }

  /**
   * Page title callback for a Popup Messages Entity  revision.
   *
   * @param int $gaya_popup_entity_revision
   *   The Popup Messages Entity  revision ID.
   *
   * @return string|TranslatableMarkup
   *   The page title.
   */
  public function revisionPageTitle(int $gaya_popup_entity_revision): string|TranslatableMarkup {
    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $gaya_popup_entity */
    $gaya_popup_entity = $this->entityTypeManager()->getStorage('gaya_popup_entity');
    $gaya_popup_entity->loadRevision($gaya_popup_entity_revision);
    return $this->t('Revision of %title from %date', [
      '%title' => $gaya_popup_entity->label(),
      '%date' => $this->dateFormatter->format($gaya_popup_entity->getRevisionCreationTime()
      ),
    ]);
  }

  /**
   * Generates an overview table of older revisions of a Popup Messages Entity .
   *
   * @param \Drupal\gaya_popup\Entity\GayaPopupInterface $gaya_popup_entity
   *   A Popup Messages Entity  object.
   *
   * @return array
   *   An array as expected by drupal_render().
   */
  public function revisionOverview(GayaPopupInterface $gaya_popup_entity): array {
    $account = $this->currentUser();
    $langcode = $gaya_popup_entity->language()->getId();
    $langname = $gaya_popup_entity->language()->getName();
    $languages = $gaya_popup_entity->getTranslationLanguages();
    $has_translations = (count($languages) > 1);
    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $gaya_popup_entity_storage */
    $gaya_popup_entity_storage = $this->entityTypeManager()->getStorage('gaya_popup_entity');

    $build['#title'] = $has_translations ? $this->t('@langname revisions for %title', [
      '@langname' => $langname,
      '%title' => $gaya_popup_entity->label(),
    ]) : $this->t('Revisions for %title', [
      '%title' => $gaya_popup_entity->label(),
    ]);
    $header = [$this->t('Revision'), $this->t('Operations')];

    $revert_permission = (($account->hasPermission("revert all popup messages entity revisions") || $account->hasPermission('administer popup messages entity entities')));
    $delete_permission = (($account->hasPermission("delete all popup messages entity revisions") || $account->hasPermission('administer popup messages entity entities')));

    $rows = [];

    $vids = $gaya_popup_entity_storage->revisionIds($gaya_popup_entity);

    $latest_revision = TRUE;

    foreach (array_reverse($vids) as $vid) {
      /** @var \Drupal\gaya_popup\Entity\GayaPopupInterface $revision */
      $revision = $gaya_popup_entity_storage->loadRevision($vid);
      // Only show revisions that are affected by the language that is being
      // displayed.
      if ($revision->hasTranslation($langcode) && $revision->getTranslation($langcode)->isRevisionTranslationAffected()) {
        $username = [
          '#theme' => 'username',
          '#account' => $revision->getRevisionUser(),
        ];

        // Use revision link to link to revisions that are not active.
        $date = $this->dateFormatter->format($revision->getRevisionCreationTime(), 'short');
        if ($vid != $gaya_popup_entity->getRevisionId()) {
          $link = Link::fromTextAndUrl($date, new Url('entity.gaya_popup_entity.revision', [
            'gaya_popup_entity' => $gaya_popup_entity->id(),
            'gaya_popup_entity_revision' => $vid,
          ]));
        }
        else {
          $link = $gaya_popup_entity->toLink($date)->toString();
        }

        $row = [];
        $column = [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '{% trans %}{{ date }} by {{ username }}{% endtrans %}{% if message %}<p class="revision-log">{{ message }}</p>{% endif %}',
            '#context' => [
              'date' => $link,
              'username' => $this->renderer->renderInIsolation($username),
              'message' => [
                '#markup' => $revision->getRevisionLogMessage(),
                '#allowed_tags' => Xss::getHtmlTagList(),
              ],
            ],
          ],
        ];
        $row[] = $column;

        if ($latest_revision) {
          $row[] = [
            'data' => [
              '#prefix' => '<em>',
              '#markup' => $this->t('Current revision'),
              '#suffix' => '</em>',
            ],
          ];
          foreach ($row as &$current) {
            $current['class'] = ['revision-current'];
          }
          $latest_revision = FALSE;
        }
        else {
          $links = [];
          if ($revert_permission) {
            $links['revert'] = [
              'title' => $this->t('Revert'),
              'url' => $has_translations ?
              Url::fromRoute('entity.gaya_popup_entity.translation_revert', [
                'gaya_popup_entity' => $gaya_popup_entity->id(),
                'gaya_popup_entity_revision' => $vid,
                'langcode' => $langcode,
              ]) :
              Url::fromRoute('entity.gaya_popup_entity.revision_revert', [
                'gaya_popup_entity' => $gaya_popup_entity->id(),
                'gaya_popup_entity_revision' => $vid,
              ]),
            ];
          }

          if ($delete_permission) {
            $links['delete'] = [
              'title' => $this->t('Delete'),
              'url' => Url::fromRoute('entity.gaya_popup_entity.revision_delete', [
                'gaya_popup_entity' => $gaya_popup_entity->id(),
                'gaya_popup_entity_revision' => $vid,
              ]),
            ];
          }

          $row[] = [
            'data' => [
              '#type' => 'operations',
              '#links' => $links,
            ],
          ];
        }

        $rows[] = $row;
      }
    }

    $build['gaya_popup_entity_revisions_table'] = [
      '#theme' => 'table',
      '#rows' => $rows,
      '#header' => $header,
    ];

    return $build;
  }

}
