<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\gaya_popup\Entity\GayaPopupInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for reverting a Popup Messages Entity revision.
 *
 * @ingroup gaya_popup
 */
class GayaPopupRevisionRevertForm extends ConfirmFormBase {


  /**
   * The Popup Messages Entity revision.
   *
   * @var \Drupal\gaya_popup\Entity\GayaPopupInterface
   */
  protected GayaPopupInterface $revision;

  /**
   * Constructs a new GayaPopupRevisionRevertForm.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $gayaPopupStorage
   *   The Popup Messages Entity storage.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    protected EntityStorageInterface $gayaPopupStorage,
    protected DateFormatterInterface $dateFormatter,
    protected TimeInterface $time,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('gaya_popup_entity'),
      $container->get('date.formatter'),
      $container->get('datetime.time')
    );
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getFormId(): string {
    return 'gaya_popup_entity_revision_revert_confirm';
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getQuestion(): TranslatableMarkup {
    return t('Are you sure you want to revert to the revision from %revision-date?', ['%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime())]);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getCancelUrl(): Url {
    return new Url('entity.gaya_popup_entity.version_history', ['gaya_popup_entity' => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getConfirmText(): TranslatableMarkup {
    return t('Revert');
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getDescription(): string|TranslatableMarkup {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function buildForm(array $form, FormStateInterface $form_state, ?int $gaya_popup_entity_revision = NULL): array {
    $this->revision = $this->gayaPopupStorage->loadRevision($gaya_popup_entity_revision);
    $form = parent::buildForm($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // The revision timestamp will be updated when the revision is saved. Keep
    // the original one for the confirmation message.
    $original_revision_timestamp = $this->revision->getRevisionCreationTime();

    $this->revision = $this->prepareRevertedRevision($this->revision, $form_state);
    $this->revision->revision_log = t('Copy of the revision from %date.', ['%date' => $this->dateFormatter->format($original_revision_timestamp)]);
    $this->revision->save();

    $this->logger('content')->notice('Popup Messages Entity: reverted %title revision %revision.',
      [
        '%title' => $this->revision->label(),
        '%revision' => $this->revision->getRevisionId(),
      ]
    );
    $this->messenger()->addStatus(t('Popup Messages Entity %title has been reverted to the revision from %revision-date.',
      [
        '%title' => $this->revision->label(),
        '%revision-date' => $this->dateFormatter->format($original_revision_timestamp),
      ])
    );
    $form_state->setRedirect(
      'entity.gaya_popup_entity.version_history',
      ['gaya_popup_entity' => $this->revision->id()]
    );
  }

  /**
   * Prepares a revision to be reverted.
   *
   * @param \Drupal\gaya_popup\Entity\GayaPopupInterface $revision
   *   The revision to be reverted.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return \Drupal\gaya_popup\Entity\GayaPopupInterface
   *   The prepared revision ready to be stored.
   */
  protected function prepareRevertedRevision(GayaPopupInterface $revision, FormStateInterface $form_state): GayaPopupInterface {
    $revision->setNewRevision();
    $revision->isDefaultRevision(TRUE);
    $revision->setRevisionCreationTime($this->time->getRequestTime());

    return $revision;
  }

}
