<?php

declare(strict_types=1);

namespace Drupal\gaya_popup\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class settings Gaya popup.
 *
 * @ingroup gaya_popup
 */
class GayaPopupSettingsForm extends ConfigFormBase {

  /**
   * Constructs a GayaPopupSettingsForm object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, protected ModuleHandlerInterface $moduleHandler) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('module_handler')
    );
  }

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The unique string identifying the form.
   */
  #[\Override]
  public function getFormId(): string {
    return 'gayapopup_settings';
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function getEditableConfigNames(): array {
    return ['gaya_popup.settings'];
  }

  /**
   * Defines the settings form for Popup Messages Entity entities.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   Form definition array.
   */
  #[\Override]
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('gaya_popup.settings');
    $form['gayapopup_settings']['#markup'] = 'Settings form for Popup Messages Entity entities. Manage field settings here.';

    $form['popup_disabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable all popups'),
      '#description' => $this->t('All Popup entities will be DISABLED when the checkbox is active.'),
      '#default_value' => $config->get('popup_disabled') ?? 0,
    ];
    $form['popup_target'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Target for ajax popup'),
      '#description' => $this->t('Allows you to choose where the ajax will insert the popup.'),
      '#default_value' => $config->get('popup_target') ?? 'body',
    ];
    $form['gaya_popup_module_css'] = [
      '#title' => t('Enable Module CSS'),
      '#type' => 'checkbox',
      '#default_value' => $config->get('gaya_popup_module_css') ?? FALSE,
      '#required' => FALSE,
    ];
    $states = [
      'visible' => [
        ':input[name="gaya_popup_module_css"]' => ['checked' => TRUE],
      ],
    ];
    $form['gaya_popup_offset'] = [
      '#title' => t('Offset'),
      '#type' => 'textfield',
      '#default_value' => $config->get('gaya_popup_offset') ?? '8rem',

      '#required' => FALSE,
      '#states' => $states,
    ];
    $form['gaya_popup_background_color'] = [
      '#title' => t('Background color'),
      '#type' => 'textfield',
      '#default_value' => $config->get('gaya_popup_background_color') ?? '#b2140f',
      '#required' => FALSE,
      '#states' => $states,
    ];
    $form['gaya_popup_text_color'] = [
      '#title' => t('Text color'),
      '#type' => 'textfield',
      '#default_value' => $config->get('gaya_popup_text_color') ?? '#fff',
      '#required' => FALSE,
      '#states' => $states,
    ];
    $form['gaya_popup_font_size'] = [
      '#title' => t('Font size'),
      '#type' => 'textfield',
      '#default_value' => $config->get('gaya_popup_font_size') ?? '1rem',
      '#required' => FALSE,
      '#states' => $states,
    ];
    if ($this->moduleHandler->moduleExists('content_translation')) {
      $form['language'] = [
        '#type' => 'details',
        '#title' => $this->t('Language settings'),
        '#open' => TRUE,
        '#tree' => TRUE,
      ];
      $form_state->set(['content_translation', 'key'], 'language');
      $form['language'] += content_translation_enable_widget('gaya_popup_entity', 'gaya_popup_entity', $form, $form_state);
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Form submission handler.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  #[\Override]
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $variables = [
      'popup_disabled' => $form_state->getValue('popup_disabled'),
      'popup_target' => $form_state->getValue('popup_target'),
      'gaya_popup_module_css' => $form_state->getValue('gaya_popup_module_css'),
      'gaya_popup_offset' => $form_state->getValue('gaya_popup_offset'),
      'gaya_popup_background_color' => $form_state->getValue('gaya_popup_background_color'),
      'gaya_popup_text_color' => $form_state->getValue('gaya_popup_text_color'),
      'gaya_popup_font_size' => $form_state->getValue('gaya_popup_font_size'),
    ];

    $config = $this->config('gaya_popup.settings');
    foreach ($variables as $key => $value) {
      $config->set($key, $value);
    }
    $config->save();
  }

}
