/**
 * @file
 * Capture, store, and auto-populate form fields with the visitor's GCLID.
 *
 * Adapted from this Google Ads code example:
 * https://support.google.com/google-ads/answer/13548778#zippy=%2Cstep-edit-your-website-to-collect-and-save-the-click-id
 */

((Drupal, drupalSettings, once) => {
  function getExpiryRecord(value) {
    const expirationDays = drupalSettings.gclid.expirationDays ?? 90;
    const expiryPeriod = expirationDays * 24 * 60 * 60 * 1000;
    const expiryDate = new Date().getTime() + expiryPeriod;

    return {
      value,
      expiryDate,
    };
  }

  function captureGclid() {
    const params = new URLSearchParams(document.location.search);
    const gclidParam = params.get("gclid");
    const gclsrcParam = params.get("gclsrc");
    const isGclsrcValid = !gclsrcParam || gclsrcParam.indexOf("aw") !== -1;

    let gclidRecord = null;
    if (gclidParam && isGclsrcValid) {
      gclidRecord = getExpiryRecord(gclidParam);
      localStorage.setItem("gclid", JSON.stringify(gclidRecord));
    }
  }

  function getGclid() {
    const gclidRecordString = localStorage.getItem("gclid");
    if (gclidRecordString) {
      const gclidRecord = JSON.parse(gclidRecordString);
      const isGclidValid =
        gclidRecord && new Date().getTime() < gclidRecord.expiryDate;
      if (isGclidValid) {
        return gclidRecord.value;
      }
    }
    return null;
  }

  function populateFields(gclid, context) {
    const gclidFormFields = drupalSettings.gclid.formFields ?? ["gclid"];
    const selectors = `input[name="${gclidFormFields.join('"], input[name="')}"]`;
    once("gclid-populate-field", selectors, context).forEach((element) => {
      element.value = gclid;
    });
  }

  Drupal.behaviors.gclid = {
    attach(context) {
      // Attempt to capture GCLID just once per page load.
      if (once("gclid-capture", document.body).length) {
        captureGclid();
      }

      // If we have a GCLID, use it to populate gclid form fields.
      const gclid = getGclid();
      if (gclid) {
        populateFields(gclid, context);
      }
    },
  };
})(Drupal, drupalSettings, once);
