<?php

namespace Drupal\gclid\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Executable\ExecutableManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure GCLID module settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Module configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $settings;

  /**
   * The user role condition.
   *
   * @var \Drupal\Core\Condition\ConditionInterface
   */
  protected $userRoleCondition;

  /**
   * SettingsForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Executable\ExecutableManagerInterface $condition_manager
   *   The plugin manager for condition plugins.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ExecutableManagerInterface $condition_manager) {
    parent::__construct($config_factory);

    $this->settings = $this->config('gclid.settings');
    $this->userRoleCondition = $condition_manager->createInstance('user_role', $this->settings->get('user_role_condition'));
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('plugin.manager.condition')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'gclid_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['gclid.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    $form['gclid_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('General settings'),
      '#open' => TRUE,
    ];
    // Convert form fields array into a multi-line string for textarea input.
    $default_form_fields_value = implode("\n", $this->settings->get('form_fields'));
    $form['gclid_settings']['form_fields'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Auto-populate form fields with these names'),
      '#description' => $this->t("Auto-populate the visitor's GCLID (if any) into any form fields on the page with these names. One field name per line."),
      '#required' => TRUE,
      '#default_value' => $default_form_fields_value,
    ];
    $form['gclid_settings']['expiration_days'] = [
      '#type' => 'number',
      '#title' => $this->t('GCLID expiration days'),
      '#description' => $this->t("How many days after the initial ad click should the GCLID remain tied to the visitor?"),
      '#required' => TRUE,
      '#default_value' => $this->settings->get('expiration_days'),
      '#min' => 1,
      '#step' => 1,
    ];

    // Add scope vertical tabs.
    $form['scope'] = [
      '#type' => 'item',
      '#title' => $this->t('Enabled for'),
    ];
    $form['scope_tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Conditions'),
      '#title_display' => 'invisible',
      '#default_tab' => 'request_path',
    ];

    // Ensure we can get at the distinct values for each subform.
    $form['#tree'] = TRUE;

    // Build and add the user_role condition configuration form.
    $condition_form = $this->userRoleCondition->buildConfigurationForm([], $form_state);
    $condition_form['#type'] = 'details';
    $condition_form['#group'] = 'scope_tabs';
    $condition_form['#title'] = $this->t('Roles');
    $form['user_role'] = $condition_form;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $this
      ->userRoleCondition
      ->validateConfigurationForm($form['user_role'], SubformState::createForSubform($form['user_role'], $form, $form_state));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this
      ->userRoleCondition
      ->submitConfigurationForm($form['user_role'], SubformState::createForSubform($form['user_role'], $form, $form_state));

    // Save configuration to settings.
    // Convert multi-line fields input into an array.
    $fields_value = $form_state->getValue(['gclid_settings', 'form_fields']);
    $fields_array = explode("\n", str_replace("\r", '', $fields_value));
    $this
      ->settings
      ->set('form_fields', $fields_array)
      ->set('expiration_days', $form_state->getValue([
        'gclid_settings',
        'expiration_days',
      ]))
      ->set('user_role_condition', $this->userRoleCondition->getConfiguration())
      ->save();
  }

}
