<?php

namespace Drupal\gcs\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Configure GCS settings for this site.
 */
class GCSSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'gcs_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['gcs.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('gcs.settings');

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Google Cloud Storage'),
      '#description' => $this->t('When enabled, all file operations will use Google Cloud Storage instead of the local filesystem.'),
      '#default_value' => $config->get('gcs.status'),
    ];

    $form['project_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google Cloud Project ID'),
      '#description' => $this->t('Your Google Cloud Platform project ID.'),
      '#default_value' => $config->get('gcs.project_id'),
      '#maxlength' => 255,
      '#required' => TRUE,
      '#states' => [
        'required' => [
          ':input[name="status"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['bucket_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Bucket Name'),
      '#description' => $this->t('The name of your Google Cloud Storage bucket.'),
      '#default_value' => $config->get('gcs.bucket_name'),
      '#maxlength' => 255,
      '#required' => TRUE,
      '#states' => [
        'required' => [
          ':input[name="status"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['authentication'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Authentication'),
      '#description' => $this->t('Configure authentication for Google Cloud Storage. You can either provide a key file path or paste the JSON key file contents.'),
    ];

    $form['authentication']['key_file'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Key File Path'),
      '#description' => $this->t('The absolute path to your Google Cloud service account JSON key file on the server. Leave empty if using key file contents below.'),
      '#default_value' => $config->get('gcs.key_file'),
      '#maxlength' => 512,
    ];

    $form['authentication']['key_file_contents'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Key File Contents'),
      '#description' => $this->t('Alternatively, paste the contents of your Google Cloud service account JSON key file here. This will be stored in the database. Leave empty if using key file path above.'),
      '#default_value' => $config->get('gcs.key_file_contents'),
      '#rows' => 10,
    ];

    $form['advanced'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Advanced Settings'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $form['advanced']['custom_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Domain'),
      '#description' => $this->t('Optional custom domain for serving files (e.g., cdn.example.com). If not set, files will be served from storage.googleapis.com.'),
      '#default_value' => $config->get('gcs.custom_domain'),
      '#maxlength' => 255,
    ];

    $form['advanced']['public_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Public Files Prefix'),
      '#description' => $this->t('Optional prefix for public:// files in the bucket (e.g., "public/").'),
      '#default_value' => $config->get('gcs.public_prefix', 'public/'),
      '#maxlength' => 255,
    ];

    $form['advanced']['private_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Private Files Prefix'),
      '#description' => $this->t('Optional prefix for private:// files in the bucket (e.g., "private/").'),
      '#default_value' => $config->get('gcs.private_prefix', 'private/'),
      '#maxlength' => 255,
    ];

    $form['advanced']['temporary_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Temporary Files Prefix'),
      '#description' => $this->t('Optional prefix for temporary:// files in the bucket (e.g., "temp/").'),
      '#default_value' => $config->get('gcs.temporary_prefix', 'temporary/'),
      '#maxlength' => 255,
    ];

    $form['warning'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning">' . $this->t('<strong>Warning:</strong> Enabling this module will redirect all file operations to Google Cloud Storage. Make sure you have properly configured authentication and that your bucket exists and is accessible.') . '</div>',
    ];

    // Show current status.
    if ($config->get('gcs.status')) {
      $status_message = $this->t('GCS is currently <strong>enabled</strong>. All file operations are using Google Cloud Storage.');
      $form['status_info'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--status">' . $status_message . '</div>',
      ];

      // Check which wrapper class is actually being used.
      $manager = \Drupal::service('stream_wrapper_manager');
      $public_wrapper = $manager->getViaUri('public://');
      $wrapper_class = get_class($public_wrapper);
      
      if (strpos($wrapper_class, 'GcsStreamWrapper') !== FALSE) {
        $form['wrapper_status'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--status">✓ Stream wrapper is active: <code>' . $wrapper_class . '</code></div>',
        ];
      }
      else {
        $form['wrapper_status'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--warning">⚠ Stream wrapper may not be active. Current class: <code>' . $wrapper_class . '</code>. Please clear all caches.</div>',
        ];
      }

      // Try to verify connection.
      try {
        $project_id = $config->get('gcs.project_id');
        $bucket_name = $config->get('gcs.bucket_name');
        if ($project_id && $bucket_name) {
          $client_config = ['projectId' => $project_id];
          if ($key_file = $config->get('gcs.key_file')) {
            $client_config['keyFilePath'] = $key_file;
          }
          elseif ($key_file_contents = $config->get('gcs.key_file_contents')) {
            $client_config['keyFile'] = json_decode($key_file_contents, TRUE);
          }

          if (!empty($client_config['keyFilePath']) || !empty($client_config['keyFile'])) {
            $storage = new \Google\Cloud\Storage\StorageClient($client_config);
            $bucket = $storage->bucket($bucket_name);
            if ($bucket->exists()) {
              $form['connection_info'] = [
                '#type' => 'markup',
                '#markup' => '<div class="messages messages--status">' . $this->t('✓ Successfully connected to bucket: @bucket', ['@bucket' => $bucket_name]) . '</div>',
              ];
            }
            else {
              $form['connection_error'] = [
                '#type' => 'markup',
                '#markup' => '<div class="messages messages--error">' . $this->t('✗ Bucket "@bucket" does not exist or is not accessible.', ['@bucket' => $bucket_name]) . '</div>',
              ];
            }
          }
          else {
            $form['connection_error'] = [
              '#type' => 'markup',
              '#markup' => '<div class="messages messages--warning">' . $this->t('Authentication not configured. Please provide either a key file path or key file contents.') . '</div>',
            ];
          }
        }
      }
      catch (\Exception $e) {
        $form['connection_error'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--error">' . $this->t('Connection error: @message', ['@message' => $e->getMessage()]) . '</div>',
        ];
      }
    }
    else {
      $form['status_info'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages">' . $this->t('GCS is currently <strong>disabled</strong>. File operations are using the local filesystem.') . '</div>',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $status = $form_state->getValue('status');

    if ($status) {
      $project_id = $form_state->getValue('project_id');
      $bucket_name = $form_state->getValue('bucket_name');
      $key_file = $form_state->getValue('key_file');
      $key_file_contents = $form_state->getValue('key_file_contents');

      // Validate that either key file path or contents is provided.
      if (empty($key_file) && empty($key_file_contents)) {
        $form_state->setError($form['authentication']['key_file'], $this->t('Either a key file path or key file contents must be provided.'));
        $form_state->setError($form['authentication']['key_file_contents'], '');
      }

      // Validate key file path if provided.
      if (!empty($key_file) && !file_exists($key_file)) {
        $form_state->setError($form['authentication']['key_file'], $this->t('The specified key file does not exist.'));
      }

      // Validate key file contents if provided.
      if (!empty($key_file_contents)) {
        $key_data = json_decode($key_file_contents, TRUE);
        if (json_last_error() !== JSON_ERROR_NONE) {
          $form_state->setError($form['authentication']['key_file_contents'], $this->t('The key file contents must be valid JSON.'));
        }
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('gcs.settings');

    $config->set('gcs.status', $form_state->getValue('status'));
    $config->set('gcs.project_id', $form_state->getValue('project_id'));
    $config->set('gcs.bucket_name', $form_state->getValue('bucket_name'));
    $config->set('gcs.key_file', $form_state->getValue('key_file'));
    $config->set('gcs.key_file_contents', $form_state->getValue('key_file_contents'));
    $config->set('gcs.custom_domain', $form_state->getValue('custom_domain'));
    $config->set('gcs.public_prefix', $form_state->getValue('public_prefix'));
    $config->set('gcs.private_prefix', $form_state->getValue('private_prefix'));
    $config->set('gcs.temporary_prefix', $form_state->getValue('temporary_prefix'));

    $config->save();

    parent::submitForm($form, $form_state);
    
    // Clear caches properly using the cache service.
    $cache_manager = \Drupal::service('cache_tags.invalidator');
    $cache_manager->invalidateTags(['config:gcs.settings']);
    
    // Force re-registration of stream wrappers.
    $wrapper_manager = \Drupal::service('stream_wrapper_manager');
    
    // Use reflection to clear all internal caches.
    $reflection = new \ReflectionClass($wrapper_manager);
    
    // Clear wrappers cache.
    if ($reflection->hasProperty('wrappers')) {
      $property = $reflection->getProperty('wrappers');
      $property->setAccessible(TRUE);
      $property->setValue($wrapper_manager, NULL);
    }
    
    // Clear schemes cache.
    if ($reflection->hasProperty('schemes')) {
      $property = $reflection->getProperty('schemes');
      $property->setAccessible(TRUE);
      $property->setValue($wrapper_manager, NULL);
    }
    
    // Clear the internal wrapper instances cache if it exists.
    if ($reflection->hasProperty('wrapperInstances')) {
      $property = $reflection->getProperty('wrapperInstances');
      $property->setAccessible(TRUE);
      $property->setValue($wrapper_manager, []);
    }
    
    // Re-register stream wrappers (this will call our hooks).
    // Don't unregister PHP wrappers here as it can cause session errors.
    $wrapper_manager->register();
    
    // Show a message instructing user to clear cache.
    if ($form_state->getValue('status')) {
      \Drupal::messenger()->addStatus($this->t('GCS configuration saved. Please <a href=":url">clear all caches</a> for the changes to take effect.', [
        ':url' => Url::fromRoute('system.performance_settings')->toString(),
      ]));
    }
    else {
      \Drupal::messenger()->addStatus($this->t('GCS has been disabled.'));
    }
  }

}
