<?php

namespace Drupal\gcs\StreamWrapper;

use Drupal\Core\StreamWrapper\StreamWrapperInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Google\Cloud\Storage\StorageClient;
use Psr\Log\LoggerInterface;

/**
 * Defines a Google Cloud Storage (GCS) stream wrapper class.
 *
 * Provides support for storing files on Google Cloud Storage.
 */
class GcsStreamWrapper implements StreamWrapperInterface {

  use StringTranslationTrait;

  /**
   * Stream context resource.
   *
   * @var resource
   */
  public $context;

  /**
   * A generic resource handle.
   *
   * @var resource
   */
  protected $handle = NULL;

  /**
   * Temporary file path for current stream operation.
   *
   * @var string
   */
  protected $tempFile = NULL;

  /**
   * Instance URI (stream).
   *
   * A stream is referenced as "scheme://target".
   *
   * @var string
   */
  protected $uri;

  /**
   * The GCS Storage Client.
   *
   * @var \Google\Cloud\Storage\StorageClient
   */
  protected static $storageClient;

  /**
   * The GCS Bucket.
   *
   * @var \Google\Cloud\Storage\Bucket
   */
  protected static $bucket;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The file scheme (public, private, etc.).
   *
   * @var string
   */
  protected $scheme;

  /**
   * Directory listing cache.
   *
   * @var array
   */
  protected static $dirCache = [];

  /**
   * Current directory position.
   *
   * @var int
   */
  protected $dirPosition = 0;

  /**
   * Constructs a new GcsStreamWrapper.
   *
   * @param \Psr\Log\LoggerInterface|null $logger
   *   The logger service. If not provided, will be retrieved from container.
   */
  public function __construct(?LoggerInterface $logger = NULL) {
    $this->logger = $logger;
  }

  /**
   * Gets the logger service.
   *
   * @return \Psr\Log\LoggerInterface
   *   The logger service.
   */
  protected function getLogger() {
    if (!isset($this->logger)) {
      $this->logger = \Drupal::service('logger.channel.gcs');
    }
    return $this->logger;
  }

  /**
   * Gets the GCS Storage Client.
   *
   * @return \Google\Cloud\Storage\StorageClient
   *   The storage client.
   */
  protected function getStorageClient() {
    if (!isset(static::$storageClient)) {
      $config = \Drupal::config('gcs.settings');
      $client_config = [
        'projectId' => $config->get('gcs.project_id'),
      ];

      // Add key file path if configured.
      if ($key_file = $config->get('gcs.key_file')) {
        $client_config['keyFilePath'] = $key_file;
      }

      // Add key file contents if configured.
      if ($key_file_contents = $config->get('gcs.key_file_contents')) {
        $client_config['keyFile'] = json_decode($key_file_contents, TRUE);
      }

      static::$storageClient = new StorageClient($client_config);
    }

    return static::$storageClient;
  }

  /**
   * Gets the GCS Bucket.
   *
   * @return \Google\Cloud\Storage\Bucket
   *   The bucket.
   */
  protected function getBucket() {
    if (!isset(static::$bucket)) {
      $config = \Drupal::config('gcs.settings');
      $bucket_name = $config->get('gcs.bucket_name');
      $storage = $this->getStorageClient();
      static::$bucket = $storage->bucket($bucket_name);
    }

    return static::$bucket;
  }

  /**
   * Gets the object name from a URI.
   *
   * @param string $uri
   *   The URI.
   *
   * @return string
   *   The object name.
   */
  protected function getObjectName($uri = NULL) {
    if (!isset($uri)) {
      $uri = $this->uri;
    }

    $target = $this->getTarget($uri);
    // Remove leading slash.
    $target = ltrim($target, '/');

    // Add scheme-specific prefix.
    $prefix = $this->getSchemePrefix($uri);
    if ($prefix && strpos($target, $prefix) !== 0) {
      $target = $prefix . $target;
    }

    return $target;
  }

  /**
   * Gets the prefix for the current scheme.
   *
   * @param string|null $uri
   *   Optional URI to extract scheme from.
   *
   * @return string
   *   The prefix, or empty string if none.
   */
  protected function getSchemePrefix($uri = NULL) {
    // Extract scheme from URI if not set.
    $scheme = $this->scheme;
    if (empty($scheme) && isset($uri)) {
      $data = explode('://', $uri, 2);
      $scheme = $data[0] ?? '';
    }
    elseif (empty($scheme) && isset($this->uri)) {
      $data = explode('://', $this->uri, 2);
      $scheme = $data[0] ?? '';
    }

    $config = \Drupal::config('gcs.settings');

    switch ($scheme) {
      case 'public':
        return $config->get('gcs.public_prefix', 'public/');

      case 'private':
        return $config->get('gcs.private_prefix', 'private/');

      case 'temporary':
        return $config->get('gcs.temporary_prefix', 'temporary/');

      default:
        return '';
    }
  }

  /**
   * Gets the target from a URI.
   *
   * @param string $uri
   *   The URI.
   *
   * @return string
   *   The target.
   */
  protected function getTarget($uri = NULL) {
    if (!isset($uri)) {
      $uri = $this->uri;
    }

    $data = explode('://', $uri, 2);
    return $data[1] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public static function getType() {
    return StreamWrapperInterface::NORMAL;
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->t('Google Cloud Storage files');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('Google Cloud Storage files stored in a GCS bucket.');
  }

  /**
   * {@inheritdoc}
   */
  public function setUri($uri) {
    $this->uri = $uri;
    // Extract scheme from URI.
    if (!empty($uri)) {
      $data = explode('://', $uri, 2);
      $this->scheme = $data[0] ?? '';
    }
    else {
      $this->scheme = '';
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getUri() {
    return $this->uri;
  }

  /**
   * {@inheritdoc}
   */
  public function getExternalUrl() {
    $config = \Drupal::config('gcs.settings');
    $bucket_name = $config->get('gcs.bucket_name');
    $object_name = $this->getObjectName();

    // Check if a custom domain is configured.
    $custom_domain = $config->get('gcs.custom_domain');
    if ($custom_domain) {
      return $custom_domain . '/' . $object_name;
    }

    // Use public URL format.
    return 'https://storage.googleapis.com/' . $bucket_name . '/' . $object_name;
  }

  /**
   * {@inheritdoc}
   */
  public function realpath() {
    // GCS doesn't have real paths, return the URI.
    return $this->uri;
  }

  /**
   * {@inheritdoc}
   */
  public function dirname($uri = NULL) {
    if (!isset($uri)) {
      $uri = $this->uri;
    }

    $target = $this->getTarget($uri);
    $dirname = dirname($target);

    if ($dirname == '.') {
      $dirname = '';
    }

    return $this->scheme . '://' . $dirname;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_open($path, $mode, $options, &$opened_path) {
    // Log that our wrapper is being called.
    \Drupal::logger('gcs')->debug('GCS stream_open called for: @path', ['@path' => $path]);
    
    $this->setUri($path);
    $object_name = $this->getObjectName();

    try {
      $bucket = $this->getBucket();

      // For reading.
      if (strpos($mode, 'r') !== FALSE) {
        if (!$bucket->object($object_name)->exists()) {
          return FALSE;
        }
        // Download to a temporary file for reading.
        $this->tempFile = tempnam(sys_get_temp_dir(), 'gcs_');
        $bucket->object($object_name)->downloadToFile($this->tempFile);
        $this->handle = fopen($this->tempFile, $mode);
        return $this->handle !== FALSE;
      }

      // For writing/appending.
      if (strpos($mode, 'w') !== FALSE || strpos($mode, 'a') !== FALSE) {
        // Create a temporary file for writing.
        $this->tempFile = tempnam(sys_get_temp_dir(), 'gcs_');
        $this->handle = fopen($this->tempFile, $mode);
        return $this->handle !== FALSE;
      }
    }
    catch (\Exception $e) {
      $this->getLogger()->error('GCS stream open failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_close() {
    if (is_resource($this->handle)) {
      $was_written = FALSE;
      $temp_file = $this->tempFile;

      // If the file was opened for writing, upload it to GCS.
      $mode = stream_get_meta_data($this->handle)['mode'];
      if ($temp_file && (strpos($mode, 'w') !== FALSE || strpos($mode, 'a') !== FALSE)) {
        try {
          $bucket = $this->getBucket();
          $object_name = $this->getObjectName();
          
          // Set ACL based on scheme.
          $options = [];
          if ($this->scheme === 'public') {
            $options['predefinedAcl'] = 'publicRead';
          }
          else {
            $options['predefinedAcl'] = 'private';
          }

          // Upload the file to GCS.
          $bucket->upload(fopen($temp_file, 'r'), [
            'name' => $object_name,
            'predefinedAcl' => $options['predefinedAcl'],
          ]);
          $was_written = TRUE;
          
          $this->getLogger()->debug('GCS file uploaded: @uri -> @object', [
            '@uri' => $this->uri,
            '@object' => $object_name,
          ]);
        }
        catch (\Exception $e) {
          $this->getLogger()->error('GCS stream close/upload failed: @message', [
            '@message' => $e->getMessage(),
          ]);
        }
      }

      fclose($this->handle);
      $this->handle = NULL;

      // Clean up temp file.
      if ($temp_file && file_exists($temp_file)) {
        unlink($temp_file);
      }
      $this->tempFile = NULL;

      return $was_written;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_read($count) {
    if (is_resource($this->handle)) {
      return fread($this->handle, $count);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_write($data) {
    if (is_resource($this->handle)) {
      return fwrite($this->handle, $data);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_eof() {
    if (is_resource($this->handle)) {
      return feof($this->handle);
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_seek($offset, $whence = SEEK_SET) {
    if (is_resource($this->handle)) {
      return fseek($this->handle, $offset, $whence) === 0;
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_tell() {
    if (is_resource($this->handle)) {
      return ftell($this->handle);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_flush() {
    if (is_resource($this->handle)) {
      return fflush($this->handle);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_stat() {
    try {
      $bucket = $this->getBucket();
      $object_name = $this->getObjectName();

      if (empty($object_name)) {
        // Root directory.
        return $this->formatDirectoryStat();
      }

      $object = $bucket->object($object_name);
      if ($object->exists()) {
        $info = $object->info();
        return $this->formatFileStat($info);
      }

      // Check if it's a directory (prefix).
      $prefix = rtrim($object_name, '/') . '/';
      $objects = $bucket->objects(['prefix' => $prefix, 'maxResults' => 1]);
      foreach ($objects as $obj) {
        return $this->formatDirectoryStat();
      }
    }
    catch (\Exception $e) {
      $this->getLogger()->error('GCS stream_stat failed: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return FALSE;
  }

  /**
   * Formats file stat information.
   *
   * @param array $info
   *   Object info from GCS.
   *
   * @return array
   *   Formatted stat array.
   */
  protected function formatFileStat(array $info) {
    $size = isset($info['size']) ? (int) $info['size'] : 0;
    $time = isset($info['timeCreated']) ? strtotime($info['timeCreated']) : time();
    $updated = isset($info['updated']) ? strtotime($info['updated']) : $time;

    return [
      'dev' => 0,
      'ino' => 0,
      'mode' => 0100666,
      'nlink' => 1,
      'uid' => 0,
      'gid' => 0,
      'rdev' => 0,
      'size' => $size,
      'atime' => $updated,
      'mtime' => $updated,
      'ctime' => $time,
      'blksize' => -1,
      'blocks' => -1,
    ];
  }

  /**
   * Formats directory stat information.
   *
   * @return array
   *   Formatted stat array.
   */
  protected function formatDirectoryStat() {
    $time = time();
    return [
      'dev' => 0,
      'ino' => 0,
      'mode' => 0040777,
      'nlink' => 1,
      'uid' => 0,
      'gid' => 0,
      'rdev' => 0,
      'size' => 0,
      'atime' => $time,
      'mtime' => $time,
      'ctime' => $time,
      'blksize' => -1,
      'blocks' => -1,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function unlink($path) {
    $this->setUri($path);
    $object_name = $this->getObjectName();

    try {
      $bucket = $this->getBucket();
      $object = $bucket->object($object_name);
      if ($object->exists()) {
        $object->delete();
        return TRUE;
      }
    }
    catch (\Exception $e) {
      $this->getLogger()->error('GCS unlink failed: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function rename($path_from, $path_to) {
    try {
      $bucket = $this->getBucket();
      $object_name_from = $this->getObjectName($path_from);
      $object_name_to = $this->getObjectName($path_to);

      $object = $bucket->object($object_name_from);
      if ($object->exists()) {
        // Copy to new location.
        $bucket->object($object_name_to)->copy($object);
        // Delete old object.
        $object->delete();
        return TRUE;
      }
    }
    catch (\Exception $e) {
      $this->getLogger()->error('GCS rename failed: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function mkdir($path, $mode, $options) {
    // GCS doesn't have directories, but we can create a placeholder object.
    // For now, we'll just return TRUE as directories are implicit in GCS.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function rmdir($path, $options) {
    $this->setUri($path);
    $object_name = $this->getObjectName();

    try {
      $bucket = $this->getBucket();
      $prefix = rtrim($object_name, '/') . '/';

      // Delete all objects with this prefix.
      $objects = $bucket->objects(['prefix' => $prefix]);
      foreach ($objects as $object) {
        $object->delete();
      }

      return TRUE;
    }
    catch (\Exception $e) {
      $this->getLogger()->error('GCS rmdir failed: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function dir_opendir($path, $options) {
    $this->setUri($path);
    $object_name = $this->getObjectName();

    // Ensure prefix ends with / for directory listing.
    $prefix = !empty($object_name) ? rtrim($object_name, '/') . '/' : '';

    try {
      $bucket = $this->getBucket();
      $list_options = ['delimiter' => '/'];
      if (!empty($prefix)) {
        $list_options['prefix'] = $prefix;
      }

      $objects = $bucket->objects($list_options);

      $files = [];
      $dirs = [];

      foreach ($objects as $object) {
        $name = $object->name();

        // Skip if this is the prefix itself (directory marker).
        if ($name === rtrim($prefix, '/')) {
          continue;
        }

        // Get relative name.
        $relative_name = !empty($prefix) ? substr($name, strlen($prefix)) : $name;

        if (empty($relative_name)) {
          continue;
        }

        // Check if it's a directory (contains /) or file.
        if (strpos($relative_name, '/') !== FALSE) {
          $dir_name = substr($relative_name, 0, strpos($relative_name, '/'));
          if (!empty($dir_name) && !in_array($dir_name, $dirs)) {
            $dirs[] = $dir_name;
          }
        }
        else {
          $files[] = $relative_name;
        }
      }

      // Also get prefixes (directories) from the delimiter.
      $prefixes = $objects->prefixes();
      foreach ($prefixes as $dir_prefix) {
        $dir_name = !empty($prefix) ? substr($dir_prefix, strlen($prefix)) : $dir_prefix;
        $dir_name = rtrim($dir_name, '/');
        if (!empty($dir_name) && !in_array($dir_name, $dirs)) {
          $dirs[] = $dir_name;
        }
      }

      static::$dirCache[$path] = array_merge($dirs, $files);
      $this->dirPosition = 0;
      return TRUE;
    }
    catch (\Exception $e) {
      $this->getLogger()->error('GCS dir_opendir failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function dir_readdir() {
    if (isset(static::$dirCache[$this->uri]) && isset(static::$dirCache[$this->uri][$this->dirPosition])) {
      return static::$dirCache[$this->uri][$this->dirPosition++];
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function dir_rewinddir() {
    $this->dirPosition = 0;
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function dir_closedir() {
    if (isset(static::$dirCache[$this->uri])) {
      unset(static::$dirCache[$this->uri]);
    }
    $this->dirPosition = 0;
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function url_stat($path, $flags) {
    $this->setUri($path);
    return $this->stream_stat();
  }

  /**
   * {@inheritdoc}
   */
  public function stream_cast($cast_as) {
    if (is_resource($this->handle)) {
      return $this->handle;
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_lock($operation) {
    // GCS doesn't support locking.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_set_option($option, $arg1, $arg2) {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_truncate($new_size) {
    if (is_resource($this->handle)) {
      return ftruncate($this->handle, $new_size);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_metadata($path, $option, $value) {
    // GCS doesn't support metadata changes like chmod, chown, etc.
    return FALSE;
  }

}
