<?php

namespace Drupal\gcs_backup\Commands;

use Drupal\gcs_backup\GcsBackupManager;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Helper\Table;

/**
 * Google cloud Backup commands.
 */
class BackupCommands extends DrushCommands {

  /**
   * Backup manager.
   *
   * @var \Drupal\gcs_backup\GcsBackupManager
   */
  private GcsBackupManager $gcsBackupManager;

  /**
   * Add dependencies.
   */
  public function __construct(
    GcsBackupManager $gcs_backup_manager,
  ) {
    parent::__construct();

    $this->gcsBackupManager = $gcs_backup_manager;
  }

  /**
   * List backups.
   */
  private function listBackups(): array {
    $backups = $this->gcsBackupManager->getBackups();
    if (empty($backups)) {
      $this->output()->writeln('No backups found.');
      return [];
    }

    $table = new Table($this->output());
    $table->setHeaders(['Index', 'Filename', 'Size', 'Date']);

    usort($backups, function ($a, $b) {
      $dateA = $a->info()['updated'] ?? '';
      $dateB = $b->info()['updated'] ?? '';
      return strtotime($dateB) - strtotime($dateA);
    });

    foreach ($backups as $index => $object) {
      $info = $object->info();
      $size = $info['size'] ?? '';
      $date = $info['updated'] ?? '';

      $table->addRow([
        $index + 1,
        $object->name(),
        $this->formatSize($size),
        $date,
      ]);
    }

    $table->render();
    return $backups;
  }

  /**
   * List backups.
   *
   * @command gcs:list
   * @aliases gcsls
   */
  public function listBackupsCommand() {
    $this->listBackups();
  }

  /**
   * List backups and download selected one.
   *
   * @command gcs:list-download
   * @aliases gcsld
   */
  public function listAndDownload() {
    $backups = $this->listBackups();
    if (empty($backups)) {
      return;
    }

    // @phpstan-ignore-next-line
    $choice = $this->io()->ask('Enter the index of the backup to download (or 0 to cancel): ');

    if ($choice > 0 && $choice <= count($backups)) {
      $selectedBackup = $backups[$choice - 1];
      try {
        $destination = $this->gcsBackupManager->downloadBackup($selectedBackup->name());
        $this->output()->writeln("Backup downloaded to: $destination");
      }
      catch (\Exception $e) {
        $this->output()->writeln('Error on backup download. Error: ' . $e->getMessage());
      }
    }
    elseif ($choice != 0) {
      $this->output()->writeln('Invalid choice. No backup downloaded.');
    }
  }

  /**
   * List backups and delete selected one.
   *
   * @command gcs:list-delete
   * @aliases gcsldel
   */
  public function listAndDelete() {
    $backups = $this->listBackups();
    if (empty($backups)) {
      return;
    }

    // @phpstan-ignore-next-line
    $choice = $this->io()->ask('Enter the index of the backup to delete (or 0 to cancel): ');

    if ($choice > 0 && $choice <= count($backups)) {
      $selectedBackup = $backups[$choice - 1];
      $select_backup_name = $selectedBackup->name();
      try {
        $selectedBackup->delete();
        $this->output()->writeln("Backup {$select_backup_name} deleted successfully.");
      }
      catch (\Exception $e) {
        $this->output()->writeln('Error on backup deletion. Error: ' . $e->getMessage());
      }
    }
    elseif ($choice != 0) {
      $this->output()->writeln('Invalid choice. No backup deleted.');
    }
  }

  /**
   * Format bytes.
   */
  private function formatSize(int $bytes): string {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= (1 << (10 * $pow));
    return round($bytes, 2) . ' ' . $units[$pow];
  }

  /**
   * Create backup and upload to Google Cloud.
   *
   * @command gcs:create-bk
   * @aliases gcscbk
   */
  public function createBackup() {
    try {
      $file_uri = $this->gcsBackupManager->createBackup();
      $this->gcsBackupManager->uploadBackupToGcs($file_uri);
    }
    catch (\Exception $e) {
      $this->output()->writeln($e->getMessage());
      return;
    }
    $this->output()->writeln('Backup created and uploaded successfully.');
  }

}
