<?php

namespace Drupal\gcs_backup;

use Drupal\Core\Config\ConfigFactoryInterface;
use Google\Cloud\Storage\Bucket;
use Drupal\Core\Database\Database;
use Drupal\Core\File\FileSystem;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Google\Cloud\Storage\StorageClient;
use Symfony\Component\Process\Process;

/**
 * Google Cloud Backup manager.
 */
class GcsBackupManager {

  /**
   * Logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Backup directory.
   *
   * @var string
   */
  protected $backupDirectory = 'private://gcs_backups/';

  /**
   * Google Cloud Storage bucket.
   *
   * @var \Google\Cloud\Storage\Bucket
   */
  protected Bucket $bucket;

  /**
   * File system service.
   *
   * @var \Drupal\Core\File\FileSystem
   */
  protected FileSystem $fileSystem;

  /**
   * Mask backup file.
   *
   * @var string
   */
  protected string $maskBackupFile = '/^.*?backup_.*\.sql\.gz$/';

  /**
   * Prefix backup.
   *
   * @var string
   */
  protected string $prefixBk;

  /**
   * Inject dependencies and set Bucket.
   */
  public function __construct(
    LoggerChannelFactoryInterface $logger_factory,
    FileSystem $file_system,
    ConfigFactoryInterface $configFactory,
  ) {
    $this->logger = $logger_factory->get('gcs_backup');
    $storage_client = new StorageClient();
    $settings = $configFactory->get('gcs_backup.settings');
    $bucket_name = $settings->get('bucket_name');
    $this->bucket = $storage_client->bucket($bucket_name);
    $this->prefixBk = $settings->get('prefix_backup') ?? '';
    $this->fileSystem = $file_system;

  }

  /**
   * Create backup locally.
   */
  public function createBackup($prefix = '', $mysqldump_path = 'mysqldump') {
    $connection_info = Database::getConnectionInfo();
    if (empty($connection_info['default'])) {
      $message = 'No default database connection is defined.';
      $this->logger->error($message);
      throw new \Exception($message);
    }
    $db_info = $connection_info['default'];

    if (empty($db_info['driver']) || ($db_info['driver'] !== 'mysql' && $db_info['driver'] !== 'mysqli')) {
      $message = 'This database backup method only supports MySQL/MariaDB.';
      $this->logger->error($message);
      throw new \Exception($message);
    }

    $database = escapeshellarg($db_info['database'] ?? '');
    $username = escapeshellarg($db_info['username'] ?? '');
    $password = escapeshellarg($db_info['password'] ?? '');
    $host     = escapeshellarg($db_info['host'] ?? '127.0.0.1');
    $port     = $db_info['port'] ?? '';

    // Build the piped command as a string.
    // Note: Check if you require port handling.
    $command = "$mysqldump_path --user=$username --password=$password --host=$host --single-transaction ";
    if (!empty($port)) {
      $command .= "--port=" . escapeshellarg($port) . " ";
    }
    $command .= "$database | gzip -9";

    // Prepare the backup file name and ensure the backup directory exists.
    $timestamp = date('Y-m-d_H-i-s');
    $filename = "backup_{$timestamp}.sql.gz";
    if ($this->prefixBk !== '') {
      $filename = $this->prefixBk . '_' . $filename;
    }
    if ($prefix !== '') {
      $filename = $prefix . '_' . $filename;
    }
    $backup_directory = $this->backupDirectory . 'upload/';
    if (!$this->fileSystem->prepareDirectory($backup_directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
      $message = 'The private file system is not properly configured or writable.';
      $this->logger->error($message);
      throw new \Exception($message);
    }
    $targetFile = $this->fileSystem->realpath($backup_directory . $filename);

    // Append redirect to file.
    $command .= ' > ' . escapeshellarg($targetFile);

    // When using pipes, we need to let the Process component
    // run the command via a shell.
    $process = Process::fromShellCommandline($command);
    // Set timeout of 30 minutes.
    // @todo , set timeout by settings.
    $process->setTimeout('1800');
    $process->run();

    if (!$process->isSuccessful()) {
      $errorOutput = $process->getErrorOutput();
      $this->logger->error('Database backup failed. mysqldump error: @output', ['@output' => $errorOutput]);
      throw new \Exception('Database backup failed. mysqldump error: ' . $errorOutput);
    }

    if (!file_exists($targetFile) || filesize($targetFile) === 0) {
      $message = 'Database backup failed. The backup file is empty.';
      $this->logger->error($message);
      throw new \Exception($message);
    }

    $this->logger->info('Database backup created at @uri', ['@uri' => $targetFile]);
    return $targetFile;
  }

  /**
   * Upload backup to Google Cloud.
   */
  public function uploadBackupToGcs(string $file_backup_uri): void {

    $normalized_backup_dir = $this->fileSystem->realpath($this->backupDirectory . 'upload/');
    $real_path = $this->fileSystem->realpath($file_backup_uri) ?: '';
    $normalized_file_dir = dirname($real_path);

    if ($normalized_backup_dir !== $normalized_file_dir) {
      $this->logger->error('Failed to validate backup file: file is not in the correct directory. Uri @uri', ['@uri' => $file_backup_uri]);
      throw new \Exception('Failed to validate backup file: File is not in the correct directory. Uri ' . $file_backup_uri);
    }

    $file_name = basename($file_backup_uri);
    if (!preg_match($this->maskBackupFile, $file_name)) {
      $this->logger->error('Failed to upload backup file to GCS, invalid uri @uri', ['@uri' => $file_backup_uri]);
      throw new \Exception('Failed to upload backup file to GCS, invalid uri: ' . $file_backup_uri);
    }

    $fileHandle = fopen($file_backup_uri, 'r');
    if ($fileHandle === FALSE) {
      $this->logger->info('Failed to open file @file', ['@file' => $file_name]);
      throw new \Exception("Failed to open file: $file_backup_uri");
    }

    try {
      $this->bucket->upload(
        $fileHandle,
        [
          'name' => $file_name,
        ]
      );
      // Delete the local file after successful upload.
      unlink($file_backup_uri);
      $this->logger->info('Backup file @file uploaded to GCS and deleted locally', ['@file' => $file_name]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to upload backup file @file to GCS. Error: @error', [
        '@file' => $file_name,
        '@error' => $e->getMessage(),
      ]);
      throw new \Exception('Failed to upload backup file to GCS: ' . $file_name . '. Error: ' . $e->getMessage());
    }
    finally {
      // Ensure the file is closed even if an exception occurs.
      if (is_resource($fileHandle)) {
        fclose($fileHandle);
      }
    }

  }

  /**
   * Returns the list of backups on Google Cloud.
   */
  public function getBackups(): array {
    $backups = [];
    $objects = $this->bucket->objects();
    foreach ($objects as $object) {
      $object_name = $object->name();

      // Only process objects that match the backup file pattern.
      if (preg_match($this->maskBackupFile, $object_name)) {
        $backups[] = $object;
      }
    }
    return $backups;
  }

  /**
   * Deletes backups following a policy.
   *
   * Remove backups after 1 year.
   * Remove backups older than 30 days except the Monday's ones.
   */
  public function applyRetentionPolicy(): void {
    $backups = $this->getBackups();

    $now = new \DateTime();
    foreach ($backups as $object) {
      $object_name = $object->name();
      $creation_time = new \DateTime($object->info()['timeCreated']);
      $days_diff = $creation_time->diff($now)->days;

      // Remove backup after 1 year.
      if ($days_diff > 365) {
        $object->delete();
        $this->logger->info('Deleted backup older than 1 year: @name', ['@name' => $object_name]);
      }
      elseif ($days_diff > 30) {
        $day_week = $creation_time->format('N');
        // Remove backups older than 30 days
        // but keep the ones done on Mondays.
        if ($day_week != 1) {
          $object->delete();
          $this->logger->info('Deleted backup older than 30 days (non-Monday): @name', ['@name' => $object_name]);
        }
      }
      // Keep all backups <= 30 days old.
    }
  }

  /**
   * Download a backup from Google Cloud.
   */
  public function downloadBackup(string $filename) {
    $file_name = basename($filename);
    if (!preg_match($this->maskBackupFile, $file_name)) {
      $this->logger->error('File @filename not valid.', ['@filename' => $filename]);
      throw new \Exception('File name not valid. File:' . $filename);
    }

    // Ensure the private directory exists.
    $privateDir = $this->backupDirectory . 'downloads/';
    $this->fileSystem->prepareDirectory($privateDir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    $object = $this->bucket->object($filename);

    if (!$object->exists()) {
      $this->logger->error('File @filename does not exist in the bucket.', ['@filename' => $filename]);
      throw new \Exception('File does not exist in the bucket. File:' . $filename);
    }

    // Download the file to the private directory.
    $destination = $privateDir . $filename;
    $real_path = $this->fileSystem->realpath($destination) ?: '';
    $object->downloadToFile($real_path);

    // Check if the file was successfully downloaded.
    if (file_exists($destination)) {
      $this->logger->info('Successfully downloaded backup: @filename', ['@filename' => $filename]);
      return $real_path;
    }

    $this->logger->error('Failed to download @filename to the private directory.', ['@filename' => $filename]);
    throw new \Exception('Failed to download file to the private directory. File: ' . $filename);
  }

}
