<?php

declare(strict_types=1);

namespace Drupal\generic_colors\Plugin\tool\Tool;

use ColorThief\ColorThief;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ConditionToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the Add Element to Webform tool.
 */
#[Tool(
  id: 'ai_agent:get_image_colors',
  label: new TranslatableMarkup('Get image colors'),
  operation: ToolOperation::Explain,
  description: new TranslatableMarkup('This method retrieves the most predominant colors of a given image.'),
  input_definitions: [
    'file_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("File ID"),
      description: new TranslatableMarkup("The file ID of a Drupal managed file."),
      required: FALSE,
    ),
    'media_id' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup("Media ID"),
      description: new TranslatableMarkup("The media ID containing an image (if provided, the plugin will try to find an image/file field and use its file)."),
      required: FALSE,
    ),
    'top_color' => new InputDefinition(
      data_type: 'boolean',
      label: new TranslatableMarkup("Only show most predominant color."),
      description: new TranslatableMarkup("If true, only the single most predominant color will be returned."),
      default_value: FALSE,
      required: FALSE,
    ),
  ]
)]
class GetImageColors extends ConditionToolBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    // Explicitly fetch services into local variables with docblocks so static
    // analyzers can infer the correct types.
    /** @var \Drupal\Core\Session\AccountInterface $current_user */
    $current_user = $container->get('current_user');

    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager */
    $entity_type_manager = $container->get('entity_type.manager');

    /** @var \Drupal\Core\File\FileSystemInterface $file_system */
    $file_system = $container->get('file_system');

    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $current_user,
    );
    $instance->entityTypeManager = $entity_type_manager;
    $instance->fileSystem = $file_system;
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    [
      'file_id' => $fid,
      'media_id' => $media_id,
      'top_color' => $top_color,
    ] = $values;

    if (!empty($fid) && !empty($media_id)) {
      return ExecutableResult::failure($this->t("Please provide only one of the id parameters (file_id or media_id)."));
    }

    if (empty($fid) && empty($media_id)) {
      return ExecutableResult::failure($this->t("Please provide one of the id parameters (file_id or media_id)."));
    }

    // Prepare file variable.
    $file = NULL;

    // If a file id is provided, try to load it directly.
    if (!empty($fid)) {
      if (!is_numeric($fid)) {
        return ExecutableResult::failure($this->t("Invalid file id."));
      }
      $file = $this->entityTypeManager->getStorage('file')->load((int) $fid);
    }

    // If a media id is provided, try to load the referenced file by scanning
    // the media fields instead of relying on a fixed field name list.
    if (empty($file) && !empty($media_id)) {
      if (!is_numeric($media_id)) {
        return ExecutableResult::failure($this->t("Invalid media id."));
      }

      $media = $this->entityTypeManager->getStorage('media')->load((int) $media_id);
      if ($media) {
        // Common field names that store file references on media entities.
        $candidate_fields = ['field_media_image', 'field_image', 'image'];
        foreach ($candidate_fields as $field_name) {
          if ($media->hasField($field_name) && !$media->get($field_name)
            ->isEmpty()) {
            $target_id = $media->get($field_name)->target_id;
            if ($target_id) {
              $file = $this->entityTypeManager->getStorage('file')
                ->load($target_id);
              if ($file) {
                break;
              }
            }
          }
        }
      }
    }

    if (!$file) {
      return ExecutableResult::failure($this->t("File not found. If you passed a Media ID instead of a File ID, ensure the media has an image/file field or pass the file's ID."));
    }

    $uri = $file->getFileUri();
    $realpath = $this->fileSystem->realpath($uri);
    if (!$realpath || !is_readable($realpath)) {
      return ExecutableResult::failure($this->t("File is not readable."));
    }

    try {
      // Get dominant color and palette.
      $dominantColor = ColorThief::getColor($realpath);
      $dominantHex = sprintf("#%02x%02x%02x", $dominantColor[0], $dominantColor[1], $dominantColor[2]);

      $palette = ColorThief::getPalette($realpath, 10);

      $colorAnalysis = $this->calculateColorPercentages($realpath, $palette);

      $result = [
        'dominant_color' => $dominantHex,
        'colors' => $colorAnalysis,
      ];

      return ExecutableResult::success($this->t('The color analysis is complete.'),
        ['result' => $top_color ? reset($colorAnalysis) : Yaml::dump($result)]);
    }
    catch (\Throwable $e) {
      return ExecutableResult::failure($this->t("I could not get the colors from image with id=@id. The error I got was: @error",
        ["@id" => ($fid ?? $media_id), "@error" => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, AccountInterface $account, bool $return_as_object = FALSE): bool|AccessResultInterface {
    // If no account is provided, use the current user.
    $account = $this->currentUser;
    $access = AccessResult::allowedIfHasPermissions($account, ['access content']);
    return $return_as_object ? $access : $access->isAllowed();
  }

  /**
   * Calculate percentage coverage for colors in the image.
   *
   * @param string $imagePath
   *   The path to the image file.
   * @param array $palette
   *   Array of RGB color arrays from ColorThief.
   *
   * @return array
   *   Array of colors with their hex values and percentage coverage.
   */
  private function calculateColorPercentages(string $imagePath, array $palette): array {
    // If ColorThief returned no palette,
    // attempt to use dominant color as single-item palette.
    if (empty($palette)) {
      try {
        $dominant = ColorThief::getColor($imagePath);
        if (is_array($dominant) && isset($dominant[0], $dominant[1], $dominant[2])) {
          $palette = [$dominant];
        }
      }
      catch (\Throwable $e) {
        // If we can't get any color, return empty results.
        return [];
      }
    }

    // Create a mapping of palette colors to their RGB arrays.
    $paletteRgb = [];
    foreach ($palette as $color) {
      $hex = sprintf("#%02x%02x%02x", $color[0], $color[1], $color[2]);
      $paletteRgb[$hex] = [$color[0], $color[1], $color[2]];
    }

    // Cache the palette keys for faster access when incrementing counts.
    $paletteKeys = array_keys($paletteRgb);

    $imageInfo = getimagesize($imagePath);
    if (!$imageInfo) {
      return [];
    }

    switch ($imageInfo['mime']) {
      case 'image/jpeg':
        $image = imagecreatefromjpeg($imagePath);
        break;

      case 'image/png':
        $image = imagecreatefrompng($imagePath);
        break;

      case 'image/gif':
        $image = imagecreatefromgif($imagePath);
        break;

      default:
        return [];
    }

    if (!$image) {
      return [];
    }

    $width = imagesx($image);
    $height = imagesy($image);

    // Aim to analyze at most this many pixels for speed. We'll downscale the
    // image first (if imagescale is available) so we can iterate every pixel
    // on the downsampled image for better spatial coverage.
    $maxSamples = 50000;
    $totalPixels = max(1, $width * $height);

    $scaled = FALSE;
    if (function_exists('imagescale') && $totalPixels > $maxSamples) {
      $scale = sqrt($maxSamples / $totalPixels);
      $newWidth = max(1, (int) floor($width * $scale));
      $newHeight = max(1, (int) floor($height * $scale));
      // Use a good quality scaling mode if available. Some PHP builds don't
      // define IMG_BILINEAR or IMG_BILINEAR_FIXED; guard against that to avoid
      // compile-time errors.
      if (defined('IMG_BILINEAR_FIXED')) {
        $resized = imagescale($image, $newWidth, $newHeight, IMG_BILINEAR_FIXED);
      }
      elseif (defined('IMG_NEAREST_NEIGHBOUR')) {
        $resized = imagescale($image, $newWidth, $newHeight, IMG_NEAREST_NEIGHBOUR);
      }
      else {
        // Fallback to calling imagescale without the mode argument (uses
        // default on systems that support it).
        $resized = imagescale($image, $newWidth, $newHeight);
      }
      if ($resized !== FALSE) {
        // Replace image resource with resized version and free original.
        if ($resized !== $image) {
          imagedestroy($image);
        }
        $image = $resized;
        $width = imagesx($image);
        $height = imagesy($image);
        $scaled = TRUE;
      }
    }

    // If we didn't scale, compute a step to sample pixels sparsely to keep the
    // number of sampled pixels under the limit.
    $step = 1;
    if (!$scaled) {
      $step = (int) floor(max(1, sqrt($totalPixels / $maxSamples)));
    }

    $colorCounts = array_fill_keys($paletteKeys, 0);

    // Iterate over sampled pixels.
    for ($x = 0; $x < $width; $x += $step) {
      for ($y = 0; $y < $height; $y += $step) {
        $rgbInt = imagecolorat($image, $x, $y);
        $colors = imagecolorsforindex($image, $rgbInt);

        // Skip fully transparent pixels (alpha == 127 in GD representation).
        if ($colors['alpha'] === 127) {
          continue;
        }

        $pixelRgb = [$colors['red'], $colors['green'], $colors['blue']];

        $closestHex = $this->findClosestColorByRgb($pixelRgb, $paletteRgb);
        if ($closestHex !== NULL) {
          $colorCounts[$closestHex]++;
        }
      }
    }

    imagedestroy($image);

    $sampledPixels = array_sum($colorCounts);
    if ($sampledPixels === 0) {
      return [];
    }

    $results = [];
    foreach ($colorCounts as $hex => $count) {
      if ($count > 0) {
        $percentage = ($count / $sampledPixels) * 100.0;
        $results[] = ['color' => $hex, 'percentage' => round($percentage, 2)];
      }
    }

    usort($results, fn($a, $b) => $b['percentage'] <=> $a['percentage']);

    return $results;
  }

  /**
   * Find the closest color in the palette to a given RGB color.
   *
   * @param array $targetRgb
   *   The target RGB color as an array.
   * @param array $paletteRgb
   *   Array of RGB colors to search in.
   *
   * @return string|null
   *   The closest hex color or null if not found.
   */
  private function findClosestColorByRgb(array $targetRgb, array $paletteRgb): ?string {
    $min = PHP_FLOAT_MAX;
    $closest = NULL;
    foreach ($paletteRgb as $hex => $rgb) {
      // Use squared distance to avoid sqrt for speed.
      $dx = $targetRgb[0] - $rgb[0];
      $dy = $targetRgb[1] - $rgb[1];
      $dz = $targetRgb[2] - $rgb[2];
      $dist2 = $dx * $dx + $dy * $dy + $dz * $dz;
      if ($dist2 < $min) {
        $min = $dist2;
        $closest = $hex;
      }
    }
    return $closest;
  }

}
