<?php

namespace Drupal\generic_colors\Plugin\AiAutomatorType;

use Drupal\ai_automators\Attribute\AiAutomatorType;
use Drupal\ai_automators\PluginBaseClasses\ExternalBase;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Tool\ToolManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * The rules for a generic colors field.
 */
#[AiAutomatorType(
  id: 'generic_image_colors',
  label: new TranslatableMarkup('Image Colors'),
  field_rule: 'string',
  target: '',
)]
class GenericColors extends ExternalBase implements AiAutomatorTypeInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'Image Colors';

  public function __construct(
    public ToolManager $toolManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('plugin.manager.tool'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function needsPrompt() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function advancedMode() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function allowedInputs() {
    return [
      'image',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function helpText() {
    return $this->t("Retrieves most popular colors in an image.");
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "This field will be automatically filled in by an AI Automator.";
  }

  /**
   * {@inheritDoc}
   *
   * @throws \Exception
   */
  public function generate($entity, $fieldDefinition, array $automatorConfig) {
    $generic_colors_tool_id = 'ai_agent:get_image_colors';
    $files = $entity->{$automatorConfig['base_field']}->getValue();
    $fid = NULL;
    if ($file = reset($files)) {
      $fid = $file['target_id'];
    }

    if (!$fid) {
      throw new NotFoundHttpException((string) new TranslatableMarkup('The Generic Colors Automator cannot access the image file.'));
    }

    if (!$this->toolManager->hasDefinition($generic_colors_tool_id)) {
      throw new NotFoundHttpException((string) new TranslatableMarkup('Tool @toolName not found.', ["@toolName" => $generic_colors_tool_id]));
    }

    $tool = $this->toolManager->createInstance($generic_colors_tool_id);
    $tool->setInputValue('file_id', $fid);
    $tool->execute();

    try {
      $result = $tool->getResult();
      if (!$result->isSuccess()) {
        throw new NotFoundHttpException('The Generic Colors Automator cannot process the image file.');
      }

      // Update node.
      $resultContextValues = $result->getContextValues();

      // Extract all color values from the nested structure:
      // $resultContextValues['colors']['colors'][INDEX]['color'].
      if (isset($resultContextValues['colors']['colors']) && is_array($resultContextValues['colors']['colors'])) {
        $field_value = array_map(function ($item) {
          if (is_array($item) && array_key_exists('color', $item)) {
            return $item['color'];
          }
          return NULL;
        }, $resultContextValues['colors']['colors']);

        // Remove any nulls and reindex the array.
        $field_value = array_values(array_filter($field_value, function ($v) {
          return $v !== NULL;
        }));
      }
      else {
        $field_value = [];
      }

      $field_name = $automatorConfig['field_name'] ?? NULL;
      if (!empty($field_name) && $entity->hasField($field_name)) {
        // Prepare items to match typical text field structure.
        $items = array_map(function ($v) {
          return ['value' => (string) $v];
        }, $field_value);

        // Set the prepared values and save.
        $entity->set($field_name, $items);
      }
    }
    catch (\Exception $e) {
      throw new \Exception("Error processing image: " . $e->getMessage());
    }

    return [];
  }

}
