<?php

namespace Drupal\gevent\Plugin\Group\Relation;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\event\Entity\EventType;
use Drupal\group\Plugin\Attribute\GroupRelationType;
use Drupal\group\Plugin\Group\Relation\GroupRelationBase;

/**
 * Provides a content enabler for events.
 */
#[GroupRelationType(
  id: 'group_event',
  entity_type_id: 'event',
  label: new TranslatableMarkup('Group event'),
  description: new TranslatableMarkup('Adds event to groups both publicly and privately.'),
  reference_label: new TranslatableMarkup('Title'),
  reference_description: new TranslatableMarkup('The title of the event to add to the group'),
  entity_access: TRUE,
  deriver: 'Drupal\gevent\Plugin\Group\Relation\GroupEventDeriver'
)]

class GroupEvent extends GroupRelationBase {

  /**
   * Retrieves the event type this plugin supports.
   *
   * @return \Drupal\event\EventTypeInterface
   *   The event type this plugin supports.
   */
  protected function getEventType() {
    return EventType::load($this->getEntityBundle());
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    $config = parent::defaultConfiguration();
    $config['entity_cardinality'] = 1;
    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Disable the entity cardinality field as the functionality of this module
    // relies on a cardinality of 1. We don't just hide it, though, to keep a UI
    // that's consistent with other content enabler plugins.
    $info = $this->t("This field has been disabled by the plugin to guarantee the functionality that's expected of it.");
    $form['entity_cardinality']['#disabled'] = TRUE;
    $form['entity_cardinality']['#description'] .= '<br /><em>' . $info . '</em>';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = parent::calculateDependencies();
    $dependencies['config'][] = 'event.type.' . $this->getRelationType()->getEntityBundle();
    return $dependencies;
  }

}
