<?php

namespace Drupal\gift_aid_commerce\Event;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\Component\EventDispatcher\Event;
use Drupal\gift_aid\Entity\CharityInterface;

/**
 * Provides a event that allows listeners to modify how a gift aid declaration should be solicited for an order.
 */
class GiftAidCommerceOrderInfoEvent extends Event {

  /**
   * An array of order items eligible for Gift Aid.
   *
   * @var \Drupal\commerce_order\Entity\OrderItemInterface[]
   */
  protected $eligible = [];

  /**
   * Whether it is recommended to solicit Gift Aid declarations for this order.
   *
   * @var bool|null
   */
  protected $recommended = NULL;

  /**
   * The charity.
   *
   * @var \Drupal\gift_aid\Entity\CharityInterface
   */
  protected $charity;

  /**
   * Constructs a gift aid related contexts event object.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   */
  public function __construct(protected readonly OrderInterface $order) {
  }

  /**
   * Gets the order.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface
   *   The order.
   */
  public function getOrder(): OrderInterface {
    return $this->order;
  }

  /**
   * Gets the specified charity to solicit a Gift Aid declaration for.
   *
   * @return \Drupal\gift_aid\Entity\CharityInterface
   *   The charity.
   */
  public function getCharity(): ?CharityInterface {
    return $this->charity;
  }

  /**
   * Specify the charity to solicit a Gift Aid declaration for.
   *
   * @param \Drupal\gift_aid\Entity\CharityInterface $charity
   *   The charity.
   *
   * @return $this
   */
  public function setCharity(CharityInterface $charity): self {
    $this->charity = $charity;
    return $this;
  }

  /**
   * Gets the order items that have been identified as eligible for Gift Aid.
   *
   * @return \Drupal\commerce_order\Entity\OrderItemInterface[]
   *   An array of order items eligible for Gift Aid.
   */
  public function getEligibleItems(): array {
    return $this->eligible;
  }

  /**
   * Specify which order items are eligible for Gift Aid.
   *
   * @param array $items
   *   The order items eligible for Gift Aid.
   *
   * @return $this
   */
  public function setEligibleItems(array $items): self {
    if (!empty(array_filter($items, function ($item) {
        return !($item instanceof OrderItemInterface && $item->getOrderId() == $this->order->id());
    }))) {
      throw new \InvalidArgumentException('All provided entities must be order items from the current order.');
    }
    $this->eligible = $items;
    return $this;
  }

  /**
   * Specify that an order item is eligible for Gift Aid.
   *
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $item
   *   The eligible order item.
   *
   * @return $this
   */
  public function addEligibleItem(OrderItemInterface $item): self {
    if ($item->getOrderId() != $this->order->id()) {
      throw new \InvalidArgumentException('The provided item must from the current order.');
    }

    // Only add the item is it is not already specified as eligible.
    if (empty(array_filter($this->eligible, function ($existingItem) use ($item) {
      return $existingItem->id() == $item->id();
    }))) {
      $this->eligible[] = $item;
    }
    return $this;
  }

  /**
   * Determines whether one or more order items are eligible for Gift Aid.
   *
   * Usually there is no need to solicit a gift aid declaration if none of the present order items are eligible.
   *
   * @return bool
   *   Whether any order item is Gift Aid eligible.
   */
  public function hasEligible(): bool {
    return !empty($this->eligible);
  }

  /**
   * Determine whether it is explicitly recommended to solicit a Gift Aid declaration for this order.
   *
   * @return bool|null
   *   Whether a declaration should be solicited. A null value indicates no opinion.
   */
  public function isRecommended(): ?bool {
    return $this->recommended;
  }

  /**
   * Specify whether it is recommended to solicit a Gift Aid declaration for this order.
   *
   * @param bool|null $recommended
   *   Whether a declaration should be solicited.
   *   A null value removes any opinion set by another subscriber.
   *
   * @return $this
   */
  public function setRecommended(?bool $recommended): self {
    $this->recommended = $recommended;
    return $this;
  }

}
