<?php

namespace Drupal\gift_aid_commerce;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\gift_aid\Entity\DeclarationInterface;

/**
 * The order declaration manager.
 */
class OrderDeclarationManager implements OrderDeclarationManagerInterface {

  /**
   * Constructs a new OrderDeclarationManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(protected readonly EntityTypeManagerInterface $entityTypeManager) {}

  /**
   * {@inheritdoc}
   */
  public function getDonor(OrderInterface $order) {
    $customer = $order->getCustomer();
    $context = $customer->isAnonymous() ? $order : $customer;
    return $this->entityTypeManager->getStorage('gift_aid_donor')->loadOrCreateByContext($context);
  }

  /**
   * {@inheritdoc}
   */
  public function declare(OrderInterface $order) {
    $charity_id = $order->getData('gift_aid_commerce_charity_id');
    assert($charity_id);

    $donor = $this->getDonor($order);

    // Only create a declaration if one does not already exist.
    if ($donor->hasOngoing()) {
      return;
    }

    if ($donor->isNew()) {
      if (($profile = $order->getBillingProfile()) && ($address = $profile->get('address')->first())) {
        $donor->setAddress($address)->save();
      }
      else {
        \Drupal::logger('gift_aid_commerce')->info("Cannot collect Gift Aid declaration for order " . $order->id() . " as address is not specified.");
        return;
      }
    }

    $fields = [
      'type' => 'commerce_order',
      'field_order' => $order,
    ];
    /** @var \Drupal\gift_aid\Entity\DeclarationStorageInterface $declaration_storage */
    $declaration_storage = $this->entityTypeManager->getStorage('gift_aid_declaration');
    $declaration = $declaration_storage->create($fields)
      ->setCharityId($charity_id)
      ->setDonor($donor)
      ->setExplanation($order->getData('gift_aid_commerce_explanation'))
      ->setValidity(DeclarationInterface::DECLARATION_INHERENTLY_VALID);
    $declaration->save();
  }

}
