<?php

namespace Drupal\gift_aid_commerce\Plugin\Commerce\CheckoutPane;

use Drupal\commerce_checkout\Attribute\CommerceCheckoutPane;
use Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface;
use Drupal\commerce_checkout\Plugin\Commerce\CheckoutPane\CheckoutPaneBase;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\gift_aid_commerce\Event\GiftAidCommerceOrderInfoEvent;
use Drupal\gift_aid_commerce\Event\GiftAidCommerceEvents;
use Drupal\gift_aid_commerce\OrderDeclarationManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Provides the gift aid information pane.
 */
#[CommerceCheckoutPane(
  id: "gift_aid_commerce",
  label: new TranslatableMarkup("Gift Aid declaration"),
  display_label: new TranslatableMarkup("Add Gift Aid"),
  default_step: "review",
)]
class GiftAidDeclaration extends CheckoutPaneBase {

  /**
   * The event that has information about how to solicit Gift Aid for this order.
   */
  protected GiftAidCommerceOrderInfoEvent $giftAidOrderInfo;

  /**
   * The event dispatcher.
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * The order service.
   */
  protected OrderDeclarationManagerInterface $orderService;

  /**
   * Whether this order is already covered by a declaration.
   *
   * @var bool
   */
  protected $covered;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, ?CheckoutFlowInterface $checkout_flow = NULL) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition, $checkout_flow);
    $instance->eventDispatcher = $container->get('event_dispatcher');
    $instance->orderService = $container->get(OrderDeclarationManagerInterface::class);
    $instance->buildOrderInfo();
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function setOrder(OrderInterface $order) {
    parent::setOrder($order);
    $this->buildOrderInfo();
    return $this;
  }

  /**
   * Build Gift Aid order info.
   */
  protected function buildOrderInfo() {
    if ($this->order) {
      $this->giftAidOrderInfo = new GiftAidCommerceOrderInfoEvent($this->order);
      $this->eventDispatcher->dispatch($this->giftAidOrderInfo, GiftAidCommerceEvents::ORDER_INFO);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'show_when' => 'always',
      'show_if_eligible' => FALSE,
      'show_if_uncovered' => TRUE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['show'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Collect Gift Aid declaration'),
    ];
    $form['show']['show_when'] = [
      '#title' => $this->t('When to collect Gift Aid declaration'),
      '#title_display' => 'invisible',
      '#type' => 'select',
      '#options' => [
        'always' => $this->t('Always'),
        'sometimes' => $this->t('Sometimes'),
      ],
      '#default_value' => $this->configuration['show_when'],
    ];
    $states = [
      'visible' => [
        ':input[name="configuration[panes][' . $this->getPluginId() . '][configuration][show][show_when]"]' => ['value' => 'sometimes'],
      ],
    ];
    $form['show']['show_if_eligible'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Only if the order contains a donation eligible for Gift Aid'),
      '#default_value' => $this->configuration['show_if_eligible'],
      '#states' => $states,
    ];
    $form['show']['show_if_uncovered'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Only if the order is not covered by a previous Gift Aid declaration'),
      '#default_value' => $this->configuration['show_if_uncovered'],
      '#states' => $states,
    ];
    $form['text'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Declaration text'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['show_when'] = $values['show']['show_when'];
      $this->configuration['show_if_eligible'] = $values['show']['show_if_eligible'];
      $this->configuration['show_if_uncovered'] = $values['show']['show_if_uncovered'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaneForm(array $pane_form, FormStateInterface $form_state, array &$complete_form) {
    $donor = $this->orderService->getDonor($this->order);
    $form_object = $this->entityTypeManager->getFormObject('gift_aid_donor', 'context');
    $form_object->setEntity($donor);
    $form_object->addGiftAidFields($pane_form, $this->giftAidOrderInfo->getCharity());
    if ($pane_form['below']['#access']) {
      // Don't show status if there is also a call-to-action.
      $pane_form['above']['status']['#access'] = FALSE;

      $pane_form['below']['checkbox'] = [
        '#type' => 'checkbox',
        '#title' => $this->t("Agree"),
        '#default_value' => FALSE,
      ];
    }

    return $pane_form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitPaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form) {
    // If the order is already placed, create the declaration now.
    // Otherwise, note it on the order for creation later when it is placed.
    // @see \Drupal\gift_aid_commerce\EventSubscriber\OrderPlaceSubscriber.
    $gift_aid_declared = $form_state->getValue(['gift_aid_commerce', 'below', 'checkbox']);
    $this->order->setData('gift_aid_commerce_declared', $gift_aid_declared);
    if ($gift_aid_declared) {
      $this->order->setData('gift_aid_commerce_charity_id', $this->giftAidOrderInfo->getCharity()->id());
      $explanation = $pane_form['below']['explanation']['contents']['#template'];
      $this->order->setData('gift_aid_commerce_explanation', $explanation);

      if ($this->order->getState()->getId() !== 'draft') {
        $this->orderService->declare($this->order);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isVisible() {
    $visible = $this->hasCharity();
    if ($visible && $this->configuration['show_when'] == 'sometimes' && $this->giftAidOrderInfo) {
      $eligible = empty($this->configuration['show_if_eligible']) || $this->giftAidOrderInfo->hasEligible();
      $uncovered = empty($this->configuration['show_if_uncovered']) || !$this->isCoveredByDeclaration();
      $visible = $this->giftAidOrderInfo->isRecommended() ?? $eligible && $uncovered;
    }
    return $visible;
  }

  /**
   * Determine whether a charity has been identified for this order.
   *
   * @return bool
   *   Whether this order has a charity.
   */
  protected function hasCharity() {
    if (!$this->giftAidOrderInfo->getCharity()) {
      \Drupal::logger('gift_aid_commerce')->warning("Cannot collect Gift Aid declaration for order " . $this->order->id() . " as charity is not specified and no default is available.");
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Determine whether this order is already covered by a declaration.
   *
   * @return bool
   *   Whether this order is already covered by a declaration.
   *
   * @throws \Exception
   */
  protected function isCoveredByDeclaration(): bool {
    if (!isset($this->covered)) {
      $this->covered = $this->orderService->getDonor($this->order)->hasOngoing($this->giftAidOrderInfo->getCharity());
      if ($this->covered) {
        \Drupal::logger('gift_aid_commerce')->info("Order " . $this->order->id() . " is already covered by a declaration for charity " . $this->giftAidOrderInfo->getCharity()->label() . ".");
      }
    }
    return $this->covered;
  }

}
