<?php

namespace Drupal\Tests\gift_aid_commerce\Functional;

use Drupal\commerce_order\Entity\Order;
use Drupal\gift_aid\Entity\Declaration;
use Drupal\gift_aid\Entity\DeclarationInterface;
use Drupal\Tests\commerce\Functional\CommerceBrowserTestBase;
use Drupal\Tests\gift_aid\Traits\GiftAidTestTrait;

/**
 * Tests the Gift Aid pane creates a declaration.
 *
 * @group gift_aid_commerce
 */
class CheckoutTest extends CommerceBrowserTestBase {

  use GiftAidTestTrait;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * The product.
   *
   * @var \Drupal\commerce_product\Entity\ProductInterface
   */
  protected $product;

  /**
   * The default profile's address.
   *
   * @var array
   */
  protected $defaultAddress = [
    'country_code' => 'US',
    'administrative_area' => 'SC',
    'locality' => 'Greenville',
    'postal_code' => '29616',
    'address_line1' => '9 Drupal Ave',
    'given_name' => 'Bryan',
    'family_name' => 'Smith',
  ];

  /**
   * The default profile.
   *
   * @var \Drupal\profile\Entity\ProfileInterface
   */
  protected $defaultProfile;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'commerce_product',
    'commerce_cart',
    'commerce_checkout',
    'gift_aid',
    'gift_aid_commerce',
    'gift_aid_commerce_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function getAdministratorPermissions() {
    return array_merge([
      'administer profile',
    ], parent::getAdministratorPermissions());
  }

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createGiftAidDefaultEntities();

    $this->store->set('billing_countries', ['FR', 'US']);
    $this->store->set('gift_aid_charity', $this->charity);
    $this->store->save();
    $this->reloadEntity($this->store);

    $variation = $this->createEntity('commerce_product_variation', [
      'type' => 'default',
      'sku' => strtolower($this->randomMachineName()),
      'price' => [
        'number' => '39.99',
        'currency_code' => 'USD',
      ],
    ]);

    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
    $this->product = $this->createEntity('commerce_product', [
      'type' => 'default',
      'title' => 'My product',
      'variations' => [$variation],
      'stores' => [$this->store],
    ]);

    $this->defaultProfile = $this->createEntity('profile', [
      'type' => 'customer',
      'uid' => $this->adminUser->id(),
      'address' => $this->defaultAddress,
    ]);

    \Drupal::state()->set('gift_aid_commerce_test_eligible', TRUE);
  }

  /**
   * Tests checkout with the Gift Aid pane.
   *
   * @dataProvider providerCollectDeclaration
   */
  public function testCollectDeclaration($isChecked) {
    $this->drupalGet($this->product->toUrl()->toString());
    $this->submitForm([], 'Add to cart');

    // Make the order require no further payment, to skip the complex payment pane.
    $order = Order::load(1);
    $order->setTotalPaid($order->getTotalPrice());
    $order->save();

    $this->drupalGet('checkout/1');
    $this->assertSession()->pageTextContains('Billing information');
    $this->assertSession()->pageTextNotContains('Payment information');

    $this->submitForm([], 'Continue to review');
    $this->assertSession()->pageTextContains('Billing information');
    $this->assertSession()->pageTextNotContains('Payment information');

    // Gift Aid pane should be present.
    $this->assertSession()->pageTextContains('Increase your donation at no extra cost to you');
    $this->assertSession()->pageTextContains('I am a UK tax payer');
    $this->assertSession()->pageTextContains('I want to Gift Aid any donations');

    $declarationStorage = \Drupal::service('entity_type.manager')->getStorage('gift_aid_declaration');
    $this->assertEmpty($declarationStorage->loadByContext($order->getCustomer()));

    $checkbox = $this->assertSession()->fieldExists('gift_aid_commerce[below][checkbox]');
    if ($isChecked) {
      $checkbox->check();
    }

    $this->submitForm([], 'Complete checkout');
    $this->assertSession()->pageTextContains('Your order number is 1. You can view your order on your account page when logged in.');

    // If the form was checked, there should now be a stored declaration.
    $declarations = $declarationStorage->loadByContext($order->getCustomer());
    $this->assertCount((int) $isChecked, $declarations);
    foreach ($declarations as $declaration) {
      $this->assertEquals($declaration->get('field_order')->target_id, $order->id());
      $this->assertStringContainsString('I am a UK taxpayer', $declaration->getExplanation());
      $this->assertEquals(Declaration::today(), $declaration->getChangedDate(TRUE));
      $this->assertEquals(DeclarationInterface::DECLARATION_INHERENTLY_VALID, $declaration->getValidity());
    }
  }

  /**
   * Provider for testCollectDeclaration().
   */
  public static function providerCollectDeclaration() {
    return [[TRUE], [FALSE]];
  }

}
