<?php

namespace Drupal\Tests\gift_aid_commerce\Kernel;

use Drupal\commerce_order\Entity\Order;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityInterface;
use Drupal\gift_aid\Declaration\DeclarationStorageInterface;
use Drupal\gift_aid\Entity\DeclarationInterface;
use Drupal\Tests\commerce_order\Kernel\OrderKernelTestBase;
use Drupal\Tests\gift_aid\Traits\GiftAidKernelTestTrait;

/**
 * Tests loading declarations in storage by context.
 *
 * @coversDefaultClass \Drupal\gift_aid\Declaration\DeclarationStorage
 * @group gift_aid_commerce
 */
class DeclarationCommerceTest extends OrderKernelTestBase {

  use GiftAidKernelTestTrait;

  /**
   * The declaration storage.
   */
  protected DeclarationStorageInterface $storage;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'datetime',
    'dynamic_entity_reference',
    'gift_aid',
    'gift_aid_commerce',
    'options',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->giftAidKernelSetUp();
    $this->storage = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration');
  }

  /**
   * Creates an order.
   *
   * @param array $values
   *   Values to set on the order.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface
   *   The order.
   */
  protected function createOrder(array $values) {
    $defaults = [
      'type' => 'default',
      'store_id' => 'test',
      'state' => 'complete',
    ];
    $values = $values + $defaults;
    $order = Order::create($values);
    $order->save();
    return $order;
  }

  /**
   * Test the LoadByContext method.
   */
  public function testContexts() {
    $this->createDeclaration();
    $this->assertDeclarationsByContextCount($this->user, 1);
    $this->assertOngoingDeclarationsByContextCount($this->user, 1);

    // Invalid declaration is not ongoing.
    $this->createDeclaration()->setValidity(DeclarationInterface::DECLARATION_INVALID)->save();
    $this->assertDeclarationsByContextCount($this->user, 2);
    $this->assertOngoingDeclarationsByContextCount($this->user, 1);

    // Declaration in the future is not ongoing.
    $this->createDeclaration()->setStartDate(new DrupalDateTime('+ 1 month'))->save();
    $this->assertDeclarationsByContextCount($this->user, 3);
    $this->assertOngoingDeclarationsByContextCount($this->user, 1);

    // Different context is counted separately.
    $order = $this->createOrder([]);
    $donor = $this->createDonor();
    $donor->setContext($order)->save();
    $this->createDeclaration()->setDonor($donor)->save();
    $this->assertDeclarationsByContextCount($order, 1);
    $this->assertDeclarationsByContextCount($this->user, 3);
  }

  /**
   * Checks the number of declarations for a context.
   *
   * @param \Drupal\Core\Entity\EntityInterface $context
   *   The entity that is the context to find declarations for.
   * @param int $count
   *   The correct number of declarations.
   */
  protected function assertDeclarationsByContextCount(EntityInterface $context, int $count) {
    $declarations = $this->storage->loadByContext($context);
    $this->assertCount($count, $declarations);
  }

  /**
   * Checks the number of ongoing declarations for a context.
   *
   * @param \Drupal\Core\Entity\EntityInterface $context
   *   The entity that is the context to find declarations for.
   * @param int $count
   *   The correct number of declarations.
   */
  protected function assertOngoingDeclarationsByContextCount(EntityInterface $context, int $count) {
    $declarations = $this->storage->loadOngoingByContext($context);
    $this->assertCount($count, $declarations);
  }

}
