<?php

namespace Drupal\Tests\gift_aid_commerce\Kernel;

use Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface;
use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\Tests\commerce_order\Kernel\OrderKernelTestBase;
use Drupal\Tests\gift_aid\Traits\GiftAidKernelTestTrait;

/**
 * Tests loading declarations in storage by context.
 *
 * @coversDefaultClass \Drupal\gift_aid\Declaration\DeclarationStorage
 * @group gift_aid_commerce
 */
class PaneVisibilityTest extends OrderKernelTestBase {

  use GiftAidKernelTestTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'datetime',
    'dynamic_entity_reference',
    'gift_aid',
    'gift_aid_commerce',
    'options',
    'user',
    'gift_aid_commerce_test',
    'commerce_checkout',
  ];

  /**
   * The user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user;

  /**
   * The order.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $order;

  /**
   * The pane manager.
   *
   * @var \Drupal\commerce_checkout\CheckoutPaneManager
   */
  protected $paneManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->giftAidKernelSetUp();
    $this->store->set('gift_aid_charity', $this->charity)->save();

    $this->order = $this->createOrder();
    $this->paneManager = $this->container->get('plugin.manager.commerce_checkout_pane');
  }

  /**
   * Test the pane is shown unconditionally if show_always is configured.
   */
  public function testShowAlways() {
    $pane = $this->getPane(['show_when' => 'always']);
    $this->assertTrue($pane->isVisible());
  }

  /**
   * Test the pane is hidden if no charity can be found.
   */
  public function testNoCharity() {
    $this->store->set('gift_aid_charity', NULL)->save();
    $this->charity->setDefault(FALSE)->save();
    $pane = $this->getPane(['show_when' => 'always']);
    $this->assertFalse($pane->isVisible());
  }

  /**
   * Test the default charity is used if none is set on store.
   */
  public function testDefaultCharity() {
    $this->store->set('gift_aid_charity', NULL)->save();
    $this->charity->setDefault()->save();
    $pane = $this->getPane(['show_when' => 'always']);
    $this->assertTrue($pane->isVisible());
  }

  /**
   * Test the pane is shown conditionally if show_when is configured.
   *
   * @dataProvider providerShowSometimes
   */
  public function testShowSometimes($show_if_uncovered, $show_if_eligible, $covered, $eligible, $expected) {
    if ($covered) {
      // Creates a declaration for user.
      $this->createDeclaration();
    }

    // See @\Drupal\gift_aid_commerce_test\EventSubscriber\GiftAidOrderStateEligibleSubscriber
    \Drupal::state()->set('gift_aid_commerce_test_eligible', $eligible);

    $pane = $this->getPane([
      'show_when' => 'sometimes',
      'show_if_uncovered' => $show_if_uncovered,
      'show_if_eligible' => $show_if_eligible,
    ]);

    $this->assertSame($expected, $pane->isVisible());
  }

  /**
   * Data provider for testShowSometimes().
   *
   * @return array
   *   An array of scenarios.
   */
  public static function providerShowSometimes() {
    return [
      [
        'show_if_uncovered' => TRUE,
        'show_if_eligible' => TRUE,
        'covered' => TRUE,
        'eligible' => TRUE,
        'expected' => FALSE,
      ],
      [
        'show_if_uncovered' => FALSE,
        'show_if_eligible' => FALSE,
        'covered' => TRUE,
        'eligible' => TRUE,
        'expected' => TRUE,
      ],
      [
        'show_if_uncovered' => FALSE,
        'show_if_eligible' => FALSE,
        'covered' => FALSE,
        'eligible' => TRUE,
        'expected' => TRUE,
      ],
      [
        'show_if_uncovered' => FALSE,
        'show_if_eligible' => FALSE,
        'covered' => TRUE,
        'eligible' => FALSE,
        'expected' => TRUE,
      ],
      [
        'show_if_uncovered' => TRUE,
        'show_if_eligible' => FALSE,
        'covered' => FALSE,
        'eligible' => FALSE,
        'expected' => TRUE,
      ],
      [
        'show_if_uncovered' => TRUE,
        'show_if_eligible' => FALSE,
        'covered' => TRUE,
        'eligible' => TRUE,
        'expected' => FALSE,
      ],
      [
        'show_if_uncovered' => FALSE,
        'show_if_eligible' => TRUE,
        'covered' => FALSE,
        'eligible' => FALSE,
        'expected' => FALSE,
      ],
    ];
  }

  /**
   * Gets a pane for testing.
   *
   * @param array $configuration
   *   Configuration for the pane.
   *
   * @return \Drupal\commerce_checkout\Plugin\Commerce\CheckoutPane\CheckoutPaneInterface
   *   The pane.
   */
  protected function getPane($configuration = []) {
    $flow = $this->createMock(CheckoutFlowInterface::class);
    $flow->expects($this->any())
      ->method('getOrder')
      ->willReturn($this->order);
    $flow->expects($this->any())
      ->method('getSteps')
      ->willReturn([]);
    return $this->paneManager->createInstance('gift_aid_commerce', $configuration, $flow);
  }

  /**
   * Creates an order.
   *
   * @param array $values
   *   Values to set on the order.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface
   *   The order.
   */
  protected function createOrder(array $values = []) {
    $defaults = [
      'type' => 'default',
      'store_id' => $this->store->id(),
      'state' => 'complete',
      'uid' => $this->user->id(),
    ];
    $values = $values + $defaults;
    $order = Order::create($values);
    $order->save();

    $item = OrderItem::create([
      'type' => 'test',
      'quantity' => '1',
    ]);
    $order->setItems([$item]);
    $order->save();

    return $order;
  }

}
