<?php

namespace Drupal\Tests\gift_aid_commerce\Unit;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\gift_aid\Entity\CharityInterface;
use Drupal\gift_aid_commerce\Event\GiftAidCommerceOrderInfoEvent;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\gift_aid_commerce\Event\GiftAidCommerceOrderInfoEvent
 * @group gift_aid_commerce
 */
class OrderInfoEventTest extends UnitTestCase {

  /**
   * An event object.
   *
   * @var \Drupal\gift_aid_commerce\Event\GiftAidCommerceOrderInfoEvent
   */
  protected $event;

  /**
   * An order.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $order;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->order = $this->createMock(OrderInterface::class);
    $this->order->expects($this->any())
      ->method('id')
      ->willReturn(mt_rand());

    $this->event = new GiftAidCommerceOrderInfoEvent($this->order);
  }

  /**
   * Creates a mock order item.
   *
   * @param string $orderId
   *   The order ID.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface
   *   The order.
   */
  protected function createMockOrderItem($orderId) {
    $item = $this->createMock(OrderItemInterface::class);
    $item->expects($this->any())
      ->method('id')
      ->willReturn(mt_rand());
    $item->expects($this->any())
      ->method('getOrderId')
      ->willReturn($orderId);
    return $item;
  }

  /**
   * Test the getOrder() method.
   */
  public function testGetOrder() {
    $this->assertSame($this->order, $this->event->getOrder());
  }

  /**
   * Test the get/set recommended methods.
   */
  public function testRecommended() {
    $this->assertNull($this->event->isRecommended());
    $this->event->setRecommended(TRUE);
    $this->assertTrue($this->event->isRecommended());
    $this->event->setRecommended(FALSE);
    $this->assertFalse($this->event->isRecommended());
    $this->event->setRecommended(NULL);
    $this->assertNull($this->event->isRecommended());
  }

  /**
   * Test get/set charity methods.
   */
  public function testCharity() {
    $charity = $this->createMock(CharityInterface::class);
    $this->event->setCharity($charity);
    $this->assertSame($charity, $this->event->getCharity());
  }

  /**
   * Test get/set/add eligible items.
   */
  public function testEligibleItems() {
    $item1 = $this->createMockOrderItem($this->order->id());
    $item2 = $this->createMockOrderItem($this->order->id());
    $items = [
      $item1,
      $item2,
    ];

    $this->assertFalse($this->event->hasEligible());

    $this->event->addEligibleItem($item1);
    $this->assertCount(1, $this->event->getEligibleItems());
    $this->assertTrue($this->event->hasEligible());
    // Duplicate items are ignored.
    $this->event->addEligibleItem($item1);
    $this->assertCount(1, $this->event->getEligibleItems());

    $this->event->addEligibleItem($item2);
    $this->assertCount(2, $this->event->getEligibleItems());
    $this->assertSame($items, $this->event->getEligibleItems());
    $this->assertTrue($this->event->hasEligible());
    // Duplicate items are ignored.
    $this->event->addEligibleItem($item2);
    $this->assertCount(2, $this->event->getEligibleItems());

    $this->event->setEligibleItems([]);
    $this->assertEmpty($this->event->getEligibleItems());
    $this->assertFalse($this->event->hasEligible());

    $this->event->setEligibleItems($items);
    $this->assertSame($items, $this->event->getEligibleItems());
    $this->assertTrue($this->event->hasEligible());

    // Items must belong to the order.
    $wrongItem = $this->createMockOrderItem($this->order->id() + 1);
    $this->event->setEligibleItems([]);
    $this->expectException(\InvalidArgumentException::class);
    $this->event->addEligibleItem($wrongItem);
    $this->expectException(\InvalidArgumentException::class);
    $this->event->setEligibleItems([$wrongItem]);
  }

}
