<?php

namespace Drupal\gift_aid\Cancellation;

use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;

/**
 * Access controller for the Cancellation entity.
 *
 * @see \Drupal\gift_aid\Entity\Cancellation.
 */
class CancellationAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\gift_aid\Entity\CancellationInterface $entity */
    $admin_permission = $this->entityType->getAdminPermission();
    switch ($operation) {
      case 'view':
        return AccessResult::allowedIfHasPermissions($account, ['view gift aid declarations', $admin_permission], 'OR');

      case 'update':
        return AccessResult::allowedIfHasPermissions($account, ['cancel gift aid declarations made by others', $admin_permission], 'OR');

      case 'delete':
        // Fallback to admin permission in parent method.
    }

    return parent::checkAccess($entity, $operation, $account);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    $admin_permission = $this->entityType->getAdminPermission();
    return AccessResult::allowedIfHasPermissions($account, ['cancel gift aid declarations made by others', $admin_permission], 'OR');
  }

  /**
   * {@inheritdoc}
   */
  protected function checkFieldAccess($operation, FieldDefinitionInterface $field_definition, AccountInterface $account, ?FieldItemListInterface $items = NULL) {
    if ($operation === 'edit') {
      $admin_permission = $this->entityType->getAdminPermission();
      $name = $field_definition->getName();

      if ($items && ($cancellation = $items->getEntity())) {
        // Anyone can set dates.
        if (in_array($name, ['date_based', 'start_date', 'end_date']) && $cancellation->isNew()) {
          return AccessResult::allowed()->addCacheableDependency($cancellation);
        }

        // Staff or admin can set any field.
        return AccessResult::allowedIfHasPermissions($account, ['cancel gift aid cancellations made by others', $admin_permission], 'OR')
          ->addCacheableDependency($cancellation);
      }
    }
    return parent::checkFieldAccess($operation, $field_definition, $account, $items);
  }

}
