<?php

namespace Drupal\gift_aid;

use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\Routing\Route;

/**
 * Checks access for the Gift Aid context form.
 */
class ContextAccess implements AccessInterface {

  /**
   * Checks access for the Gift Aid context form.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match object.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account to check access for.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   An access result object.
   */
  public function access(Route $route, RouteMatchInterface $route_match, AccountInterface $account) {
    // If no charities are defined, deny access.
    $charity_ids = \Drupal::entityTypeManager()->getStorage('gift_aid_charity')->getQuery()->accessCheck(FALSE)->execute();
    if (empty($charity_ids)) {
      return AccessResult::forbidden()->addCacheTags(['gift_aid_charity_list']);
    }

    $params = $route_match->getParameters()->all();
    /** @var \Drupal\Core\Entity\EntityInterface $context */
    $context = reset($params);

    // Load the donor entity from the context provided.
    /** @var \Drupal\gift_aid\Donor\DonorStorageInterface $storage */
    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_donor');
    $donor = $storage->loadOrCreateByContext($context);

    switch ($route->getRequirement('_gift_aid_context')) {
      case 'declare':
        $admin_perm = 'record gift aid declarations made by others';
        $own_perm = 'make gift aid declaration';
        break;

      case 'cancel':
        $admin_perm = 'cancel gift aid declarations made by others';
        $own_perm = 'cancel gift aid declaration';
        break;

      default:
        return AccessResult::neutral()->addCacheTags(['gift_aid_charity_list']);
    }

    $own_result = AccessResult::allowedIf($donor->isCurrentUser($account))->addCacheContexts(['user'])
      ->andIf(AccessResult::allowedIfHasPermission($account, $own_perm));

    return $own_result->orIf(AccessResult::allowedIfHasPermission($account, $admin_perm))
      ->addCacheTags(['gift_aid_charity_list']);
  }

}
