<?php

namespace Drupal\gift_aid\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * View donor for a context, for example as a tab on a user.
 */
class ContextOverviewController extends ControllerBase {

  /**
   * Overview page for a donor context.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array
   *   A render array.
   */
  public function view(Request $request) {
    $donor = $this->getDonor($request);

    if (!$donor->isCurrentUser() && !$donor->isNew()) {
      // For admin of existing donor, editing the entity is not the mainline
      // case (mostly for changing the address), so view the entity, with an
      // edit button if needed.
      $view_builder = $this->entityTypeManager()->getViewBuilder('gift_aid_donor');
      return $view_builder->view($donor, 'context');
    }
    else {
      // Show the donor form with a wrapper.
      return $this->entityFormBuilder()->getForm($donor, 'context');
    }
  }

  /**
   * Cancellation page for a donor context.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array
   *   A render array.
   */
  public function cancel(Request $request) {
    $donor = $this->getDonor($request);

    if (!$donor->isCurrentUser()) {
      // @todo For admin, show a form for creating a cancellation.
      return NULL;
    }
    else {
      // Show the cancel confirm form.
      return $this->entityFormBuilder()->getForm($donor, 'cancel');
    }
  }

  /**
   * Gets the donor entity associated with a request.
   */
  protected function getDonor(Request $request) {
    // Find the parameter that is an entity - we don't know its name.
    $params = array_filter($request->attributes->all(), fn($p) => $p instanceof EntityInterface);
    /** @var \Drupal\Core\Entity\EntityInterface $context */
    $context = reset($params);

    // Load the donor entity from the context provided.
    /** @var \Drupal\gift_aid\Donor\DonorStorageInterface $storage */
    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_donor');
    return $storage->loadOrCreateByContext($context);
  }

}
