<?php

namespace Drupal\gift_aid\Declaration;

use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\gift_aid\Entity\DeclarationInterface;

/**
 * Access controller for the Declaration entity.
 *
 * @see \Drupal\gift_aid\Entity\Declaration.
 */
class DeclarationAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\gift_aid\Entity\DeclarationInterface $entity */
    $admin_permission = $this->entityType->getAdminPermission();
    switch ($operation) {
      case 'view':
        return AccessResult::allowedIfHasPermissions($account, ['view gift aid declarations', $admin_permission], 'OR');

      case 'update':
        return AccessResult::allowedIfHasPermissions($account, ['record gift aid declarations made by others', $admin_permission], 'OR');

      case 'delete':
        // Fallback to admin permission in parent method.
    }

    return parent::checkAccess($entity, $operation, $account);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    $type = \Drupal::entityTypeManager()->getStorage($this->entityType->getBundleEntityType())->load($entity_bundle);
    if ($type->getMethod() == 'web') {
      return AccessResult::forbidden()->addCacheableDependency($type);
    }

    $admin_permission = $this->entityType->getAdminPermission();
    $specific_permission = "create $entity_bundle declarations";

    return AccessResult::allowedIfHasPermissions($account, ['record gift aid declarations made by others', $specific_permission])
      ->orIf(AccessResult::allowedIfHasPermission($account, $admin_permission));
  }

  /**
   * {@inheritdoc}
   */
  protected function checkFieldAccess($operation, FieldDefinitionInterface $field_definition, AccountInterface $account, ?FieldItemListInterface $items = NULL) {
    if ($operation === 'edit' && $items && ($declaration = $items->getEntity())) {
      $admin_permission = $this->entityType->getAdminPermission();
      $name = $field_definition->getName();

      if ($declaration->isLocked() && in_array($name, DeclarationInterface::CRITICAL_FIELDS)) {
        // Only admin can set critical fields when locked.
        return AccessResult::allowedIfHasPermission($account, $admin_permission)
          ->addCacheableDependency($declaration);
      }

      if (in_array($name, ['confirmation_date', 'changed_date', 'cancellation_date'])) {
        // Only admin can set fields that are normally set programmatically.
        return AccessResult::allowedIfHasPermission($account, $admin_permission)
          ->addCacheableDependency($declaration);
      }
    }

    return parent::checkFieldAccess($operation, $field_definition, $account, $items);
  }

}
