<?php

namespace Drupal\gift_aid\Entity;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;

/**
 * Defines the Cancellation entity.
 *
 * @ingroup gift_aid
 *
 * @ContentEntityType(
 *   id = "gift_aid_cancellation",
 *   label = @Translation("Gift Aid cancellation"),
 *   label_collection = @Translation("Gift Aid cancellations"),
 *   label_singular = @Translation("Gift Aid cancellation"),
 *   label_plural = @Translation("Gift Aid cancellations"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Gift Aid cancellation",
 *     plural = "@count Gift Aid cancellations"
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\gift_aid\Record\RecordListBuilder",
 *     "form" = {
 *       "default" = "Drupal\gift_aid\Form\RecordForm",
 *       "delete" = "Drupal\gift_aid\Form\GiftAidDeleteForm",
 *     },
 *     "access" = "Drupal\gift_aid\Cancellation\CancellationAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider",
 *       "revision" = "Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "gift_aid_cancellation",
 *   data_table = "gift_aid_cancellation_field_data",
 *   revision_table = "gift_aid_cancellation_revision",
 *   revision_data_table = "gift_aid_cancellation_field_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer gift aid cancellations",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "uuid" = "uuid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log"
 *   },
 *   links = {
 *     "canonical" = "/admin/gift-aid/cancellation/{gift_aid_cancellation}",
 *     "add-form" = "/admin/gift-aid/cancellation/add",
 *     "edit-form" = "/admin/gift-aid/cancellation/{gift_aid_cancellation}/edit",
 *     "delete-form" = "/admin/gift-aid/cancellation/{gift_aid_cancellation}/delete",
 *     "collection" = "/admin/gift-aid/cancellation",
 *     "version-history" = "/admin/gift-aid/cancellation/{gift_aid_cancellation}/revisions",
 *     "revision" = "/admin/gift-aid/cancellation/{gift_aid_cancellation}/revisions/{gift_aid_cancellation_revision}/view",
 *   },
 *   constraints = {
 *     "GiftAidStartDate" = {}
 *   }
 * )
 */
class Cancellation extends RecordBase implements RecordInterface {

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);

    $this->notify();
    if ($this->original && $this->original->getDonorId() != $this->getDonorId()) {
      $this->original->notify();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return Html::escape("{$this->getDonor()->label()} {$this->getDeclaredDate(TRUE)} ({$this->id()})");
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['donor']->setDescription(t('The donor for the cancellation.'));
    $fields['charity']->setDescription(t('The charity the cancellation is for.'));
    $fields['evidence']->setLabel(t('Evidence'))
      ->setDescription(t('Evidence for the cancellation.'))
      ->setRequired(TRUE);
    $fields['declared_date']->setDescription(t('The date the cancellation was made by the donor.'));
    $fields['date_based']->setDescription(t('Is this a date-based cancellation that covers all donations within a period of time?'));
    $fields['start_date']->setDescription(t('Only donations made after this date are covered by the cancellation.'));
    $fields['end_date']->setDescription(t('Donations made after this date are not covered by the cancellation.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public static function postDelete(EntityStorageInterface $storage, array $entities) {
    foreach ($entities as $cancellation) {
      $cancellation->notify();
    }
  }

  /**
   * Notifies all relevant declarations about the new or updated cancellation.
   */
  protected function notify() {
    foreach ($this->getDonor()->getDeclarations() as $declaration) {
      $declaration->calculateCancellation();
    }
  }

}
