<?php

namespace Drupal\gift_aid\Entity;

use CommerceGuys\Addressing\AddressFormat\AddressField;
use CommerceGuys\Addressing\AddressFormat\FieldOverride;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;

/**
 * Defines the Charity entity.
 *
 * @ingroup gift_aid
 *
 * @ContentEntityType(
 *   id = "gift_aid_charity",
 *   label = @Translation("Charity"),
 *   plural_label = @Translation("Charities"),
 *   label = @Translation("Charity"),
 *   label_collection = @Translation("Charities"),
 *   label_singular = @Translation("charity"),
 *   label_plural = @Translation("charities"),
 *   label_count = @PluralTranslation(
 *     singular = "@count charity",
 *     plural = "@count charities"
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\gift_aid\Charity\CharityListBuilder",
 *     "storage" = "Drupal\gift_aid\Charity\CharityStorage",
 *     "form" = {
 *       "default" = "Drupal\gift_aid\Form\Charity\CharityForm",
 *       "add" = "Drupal\gift_aid\Form\Charity\CharityForm",
 *       "edit" = "Drupal\gift_aid\Form\Charity\CharityForm",
 *       "delete" = "Drupal\gift_aid\Form\Charity\CharityDeleteForm",
 *     },
 *     "access" = "Drupal\gift_aid\Charity\CharityAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\gift_aid\Charity\CharityHtmlRouteProvider",
 *       "revision" = "Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "gift_aid_charity",
 *   data_table = "gift_aid_charity_field_data",
 *   revision_table = "gift_aid_charity_revision",
 *   revision_data_table = "gift_aid_charity_field_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "configure gift aid",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "langcode" = "langcode",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log"
 *   },
 *   links = {
 *     "canonical" = "/admin/gift-aid/charity/{gift_aid_charity}",
 *     "add-form" = "/admin/gift-aid/charity/add",
 *     "edit-form" = "/admin/gift-aid/charity/{gift_aid_charity}/edit",
 *     "delete-form" = "/admin/gift-aid/charity/{gift_aid_charity}/delete",
 *     "collection" = "/admin/gift-aid/charity",
 *     "version-history" = "/admin/gift-aid/charity/{gift_aid_charity}/revisions",
 *     "revision" = "/admin/gift-aid/charity/{gift_aid_charity}/revisions/{gift_aid_charity_revision}/view",
 *   },
 *   field_ui_base_route = "entity.gift_aid_charity.collection"
 * )
 */
class Charity extends RevisionableContentEntityBase implements CharityInterface {

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRegisteredName() {
    return $this->get('registered_name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setRegisteredName($registered_name) {
    $this->set('registered_name', $registered_name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRegistrationNumber() {
    return $this->get('registration_number')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setRegistrationNumber($registration_number) {
    $this->set('registration_number', $registration_number);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getExplanation() {
    return $this->get('explanation')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setExplanation($explanation) {
    $this->set('explanation', $explanation);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRegisteredAddress() {
    return $this->get('registered_address')->first();
  }

  /**
   * {@inheritdoc}
   */
  public function setRegisteredAddress($registered_address) {
    $this->set('registered_address', $registered_address);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isDefault() {
    return (bool) $this->get('is_default')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setDefault($is_default = TRUE) {
    $this->set('is_default', (bool) $is_default);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name the charity is usually known by.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    // Registered Name field.
    $fields['registered_name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Registered Name'))
      ->setDescription(t('The Registered Name of the Charity.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    // Registration Number field.
    $fields['registration_number'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Registration Number'))
      ->setDescription(t('The Registration Number of the Charity.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['explanation'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Explanation'))
      ->setDescription(t('The explanation to show to the donor in accordance with HMRC guidance. This field accepts placeholders in the form of Twig variables.'))
      ->setDefaultValue(t("I want to Gift Aid any donations I make from now on or have made in the past four years to {{ charity.label }}. I am a UK taxpayer and understand that if I pay less Income Tax and/or Capital Gains Tax than the amount of Gift Aid claimed on all my donations in that tax year it is my responsibility to pay any difference."))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    // Registered Address field.
    $fields['registered_address'] = BaseFieldDefinition::create('address')
      ->setLabel(t('Registered Address'))
      ->setDescription(t('The Registered Address of the Charity.'))
      ->setSetting('field_overrides', [
        AddressField::GIVEN_NAME => ['override' => FieldOverride::HIDDEN],
        AddressField::ADDITIONAL_NAME => ['override' => FieldOverride::HIDDEN],
        AddressField::FAMILY_NAME => ['override' => FieldOverride::HIDDEN],
        AddressField::ORGANIZATION => ['override' => FieldOverride::HIDDEN],
        AddressField::ADDRESS_LINE3 => ['override' => FieldOverride::HIDDEN],
      ])
      ->setDisplayOptions('form', [
        'type' => 'address_default',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // 'default' is a reserved SQL word, hence the 'is_' prefix.
    $fields['is_default'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Default'))
      ->setDescription(t('Whether this is the default charity.'))
      ->setDefaultValue(FALSE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => TRUE,
        ],
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayConfigurable('form', TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);
    // The first charity is the default.
    if ($this->isNew() && !$this->isDefault()) {
      $charity_ids = $storage->getQuery()
        ->condition('is_default', TRUE)
        ->accessCheck(FALSE)
        ->execute();
      if (empty($charity_ids)) {
        $this->setDefault();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    /** @var \Drupal\gift_aid\Charity\CharityStorage $storage */
    parent::postSave($storage, $update);

    $default = $this->isDefault();
    $original_default = $this->original ? $this->original->isDefault() : FALSE;
    if ($default && !$original_default) {
      // The charity was set as default, remove the flag from other charities.
      $charity_ids = $storage->getQuery()
        ->condition('id', $this->id(), '<>')
        ->condition('is_default', TRUE)
        ->accessCheck(FALSE)
        ->execute();
      foreach ($charity_ids as $charity_id) {
        /** @var \Drupal\gift_aid\Entity\CharityInterface $charity */
        $charity = $storage->load($charity_id);
        $charity->setDefault(FALSE);
        $charity->save();
      }
    }
  }

}
