<?php

namespace Drupal\gift_aid\Entity;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the Declaration entity.
 *
 * @ingroup gift_aid
 *
 * @ContentEntityType(
 *   id = "gift_aid_declaration",
 *   label = @Translation("Gift Aid declaration"),
 *   label_collection = @Translation("Gift Aid declarations"),
 *   label_singular = @Translation("Gift Aid declaration"),
 *   label_plural = @Translation("Gift Aid declarations"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Gift Aid declaration",
 *     plural = "@count Gift Aid declarations"
 *   ),
 *   bundle_label = @Translation("Declaration type"),
 *   handlers = {
 *     "list_builder" = "Drupal\gift_aid\Record\RecordListBuilder",
 *     "storage" = "Drupal\gift_aid\Declaration\DeclarationStorage",
 *     "views_data" = "Drupal\gift_aid\Declaration\DeclarationViewsData",
 *     "form" = {
 *       "default" = "Drupal\gift_aid\Form\RecordForm",
 *       "delete" = "Drupal\gift_aid\Form\GiftAidDeleteForm",
 *     },
 *     "access" = "Drupal\gift_aid\Declaration\DeclarationAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\gift_aid\Declaration\DeclarationRouteProvider",
 *       "revision" = "Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "gift_aid_declaration",
 *   data_table = "gift_aid_declaration_field_data",
 *   revision_table = "gift_aid_declaration_revision",
 *   revision_data_table = "gift_aid_declaration_field_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer gift aid declarations",
 *   entity_keys = {
 *     "id" = "id",
 *     "bundle" = "type",
 *     "revision" = "revision_id",
 *     "uuid" = "uuid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log"
 *   },
 *   links = {
 *     "canonical" = "/admin/gift-aid/declaration/{gift_aid_declaration}",
 *     "add-page" = "/admin/gift-aid/declaration/add",
 *     "add-form" = "/admin/gift-aid/declaration/add/{gift_aid_declaration_type}",
 *     "edit-form" = "/admin/gift-aid/declaration/{gift_aid_declaration}/edit",
 *     "delete-form" = "/admin/gift-aid/declaration/{gift_aid_declaration}/delete",
 *     "collection" = "/admin/gift-aid/declaration",
 *     "version-history" = "/admin/gift-aid/declaration/{gift_aid_declaration}/revisions",
 *     "revision" = "/admin/gift-aid/declaration/{gift_aid_declaration}/revisions/{gift_aid_declaration_revision}/view",
 *   },
 *   bundle_entity_type = "gift_aid_declaration_type",
 *   field_ui_base_route = "entity.gift_aid_declaration_type.edit_form",
 *   constraints = {
 *     "GiftAidStartDate" = {}
 *   }
 * )
 */
class Declaration extends RecordBase implements DeclarationInterface {

  /**
   * Declaration becomes locked when last critical change at most this time.
   */
  const STABILITY_WINDOW = '7 days ago';

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    $original = $this->getOriginal();

    foreach (static::CRITICAL_FIELDS as $field) {
      if (!$original || !$this->get($field)->equals($original->get($field))) {
        $this->setChangedDate();
        break;
      }
    }

    parent::preSave($storage);
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return Html::escape("{$this->getDonor()->label()} {$this->getDeclaredDate(TRUE)} {$this->type->entity->label()} ({$this->id()})");
  }

  /**
   * {@inheritdoc}
   */
  public function getExplanation() {
    return $this->get('explanation')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setExplanation(string $explanation) {
    $this->set('explanation', $explanation);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateCancellation() {
    $result = NULL;

    foreach ($this->getDonor()->getCancellations() as $cancellation) {
      if ($cancellation->getDeclaredDate(TRUE) < $this->getDeclaredDate(TRUE)) {
        // Declaration is more recent than cancellation, so they changed their
        // mind.
        continue;
      }

      if ($cancellation->hasEndDate() && $cancellation->getEndDate(TRUE) < $this->getStartDate(TRUE)) {
        // Cancellation ends before declaration starts.
        continue;
      }

      if ($this->hasEndDate() && $this->getEndDate(TRUE) < $cancellation->getStartDate(TRUE)) {
        // Cancellation starts after declaration ends.
        continue;
      }

      $date = $cancellation->getStartDate(TRUE);
      if (!$result || $result > $date) {
        $result = $date;
      }
    }

    $this->setDateField('cancellation_date', $result)->save();
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCancellationDate(bool $formatted = FALSE) {
    return $this->getDateField('cancellation_date', $formatted);
  }

  /**
   * {@inheritdoc}
   */
  public function hasCancellationDate() {
    return !$this->get('cancellation_date')->isEmpty();
  }

  /**
   * {@inheritdoc}
   */
  public function isCancelled() {
    $date = $this->getCancellationDate(TRUE);
    return $date && ($date <= static::today());
  }

  /**
   * {@inheritdoc}
   */
  public function isConfirmationSent() {
    return !$this->get('confirmation_date')->isEmpty();
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmationDate(bool $formatted = FALSE) {
    return $this->getDateField('confirmation_date', $formatted);
  }

  /**
   * {@inheritdoc}
   */
  public function setConfirmationDate($date) {
    return $this->setDateField('confirmation_date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function getChangedDate(bool $formatted = FALSE) {
    return $this->getDateField('changed_date', $formatted);
  }

  /**
   * {@inheritdoc}
   */
  public function setChangedDate($date = NULL) {
    $date ??= static::today();
    return $this->setDateField('changed_date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function isLocked() {
    return !$this->isNew() && $this->getChangedDate(TRUE) <= static::today(static::STABILITY_WINDOW);
  }

  /**
   * {@inheritdoc}
   */
  public function isClaimable() {
    return $this->isValid() && $this->isLocked();
  }

  /**
   * {@inheritdoc}
   */
  public function isValid() {
    return $this->getValidity() == DeclarationInterface::DECLARATION_INHERENTLY_VALID ||
      ($this->getValidity() == DeclarationInterface::DECLARATION_VALID_IF_CONFIRMED && $this->isConfirmationSent());
  }

  /**
   * {@inheritdoc}
   */
  public function isConfirmationMissing() {
    return $this->getValidity() == DeclarationInterface::DECLARATION_VALID_IF_CONFIRMED && !$this->isConfirmationSent();
  }

  /**
   * {@inheritdoc}
   */
  public function getValidity() {
    return $this->get('validity')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setValidity(string $valid) {
    $this->set('validity', $valid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus() {
    // @todo provide a documented and centralized source of truth for declaration status semantics.
    if ($this->getValidity() == DeclarationInterface::DECLARATION_INVALID) {
      return DeclarationInterface::DECLARATION_INVALID;
    }
    elseif ($this->isCancelled()) {
      return DeclarationInterface::DECLARATION_CANCELLED;
    }
    elseif (!$this->isValid()) {
      return DeclarationInterface::DECLARATION_CONFIRMATION_MISSING;
    }
    elseif (!$this->isClaimable()) {
      return DeclarationInterface::DECLARATION_PROVISIONAL;
    }
    elseif (!$this->isDateBased()) {
      return DeclarationInterface::DECLARATION_SELECTIVE;
    }
    elseif ($this->hasEnded()) {
      return DeclarationInterface::DECLARATION_ENDED;
    }
    elseif ($this->hasStarted()) {
      return DeclarationInterface::DECLARATION_ONGOING;
    }
    else {
      return DeclarationInterface::DECLARATION_PENDING;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusString() {
    return static::statusOptions()[$this->getStatus()];
  }

  /**
   * {@inheritdoc}
   */
  public function isOngoing() {
    return $this->isValid() && $this->hasStarted() && !$this->hasEnded() && !$this->isCancelled();
  }

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return $this->bundle();
  }

  /**
   * {@inheritdoc}
   */
  public static function validityOptions() {
    return [
      DeclarationInterface::DECLARATION_INVALID => t('Invalid'),
      DeclarationInterface::DECLARATION_VALID_IF_CONFIRMED => t('Valid if confirmed'),
      DeclarationInterface::DECLARATION_INHERENTLY_VALID => t('Inherently valid'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function statusOptions() {
    return [
      DeclarationInterface::DECLARATION_INVALID => t('Invalid'),
      DeclarationInterface::DECLARATION_CANCELLED => t('Cancelled'),
      DeclarationInterface::DECLARATION_CONFIRMATION_MISSING => t('Written confirmation missing'),
      DeclarationInterface::DECLARATION_PROVISIONAL => t('Provisional'),
      DeclarationInterface::DECLARATION_SELECTIVE => t('Selective'),
      DeclarationInterface::DECLARATION_PENDING => t('Pending'),
      DeclarationInterface::DECLARATION_ONGOING => t('Ongoing'),
      DeclarationInterface::DECLARATION_ENDED => t('Ended'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['explanation'] = BaseFieldDefinition::create('string')
      ->setSetting('max_length', 1000)
      ->setLabel(t('Explanation'))
      ->setDescription(t('Explanation shown to the donor of the personal tax implications associated with making a Gift Aid donation.'))
      ->setDisplayOptions('form', ['region' => 'hidden'])
      ->setDisplayOptions('view', ['weight' => 0])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $valid_description = t('A valid declaration must: <ul><li>state the donor’s full name and home address</li><li>name the charity</li><li>identify the gift or ifts to which the declaration relates (for example, a particular donation or all donations)</li><li>confirm that the identified gift or gifts are to be treated as Gift Aid donations</li><li>demonstrate that the Charity has given an adequate explanation to the donor of the personal tax implications</li></ul>');

    $fields['validity'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Validity'))
      ->setDescription($valid_description)
      ->setDefaultValue(DeclarationInterface::DECLARATION_INVALID)
      ->setRevisionable(TRUE)
      ->setSetting('allowed_values', static::validityOptions())
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_buttons',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['confirmation_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Written confirmation sent date'))
      ->setDescription(t('The date a written confirmation of the declaration was most recently sent to the donor. This is required for oral declarations, and can also be used as part of a process of validating declarations that were invalid due to insufficient auditable information.'))
      ->setRevisionable(TRUE)
      ->setSetting('datetime_type', 'date')
      ->setDisplayOptions('view', [
        'type' => 'datetime_default',
        'settings' => [
          'format_type' => 'html_date',
        ],
        'label' => 'inline',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_default',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Changed date'))
      ->setDescription(t('The date when the declaration last had a significant change that affects Gift Aid eligibility.'))
      ->setRevisionable(TRUE)
      ->setSetting('datetime_type', 'date')
      ->setDisplayOptions('view', [
        'type' => 'datetime_default',
        'label' => 'inline',
        'weight' => 0,
        'settings' => [
          'format_type' => 'html_date',
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_default',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['cancellation_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Cancellation date'))
      ->setDescription(t('The date the declaration is cancelled from.'))
      ->setRevisionable(TRUE)
      ->setSetting('datetime_type', 'date')
      ->setDisplayOptions('view', [
        'type' => 'datetime_default',
        'label' => 'inline',
        'weight' => 0,
        'settings' => [
          'format_type' => 'html_date',
        ],
      ])
      ->setDisplayOptions('form', ['region' => 'hidden'])
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

}
