<?php

namespace Drupal\gift_aid\Entity;

/**
 * Provides an interface for defining declarations.
 *
 * @ingroup gift_aid
 */
interface DeclarationInterface extends RecordInterface {

  // Declaration validity.
  const DECLARATION_INVALID = 'invalid';
  const DECLARATION_VALID_IF_CONFIRMED = 'valid if confirmed';
  const DECLARATION_INHERENTLY_VALID = 'inherently valid';

  // Declaration status constants (in addition to DECLARATION_INVALID).
  const DECLARATION_CANCELLED = 'cancelled';
  const DECLARATION_CONFIRMATION_MISSING = 'confirmation missing';
  const DECLARATION_PROVISIONAL = 'provisional';
  const DECLARATION_SELECTIVE = 'limited';
  const DECLARATION_PENDING = 'pending';
  const DECLARATION_ONGOING = 'ongoing';
  const DECLARATION_ENDED = 'ended';

  /**
   * List of critical fields.
   *
   * Altering these causes an update of the "changed" field, and these cannot
   * be changed when the declaration is locked.
   */
  const CRITICAL_FIELDS = [
    'donor',
    'charity',
    'declared_date',
    'date_based',
    'start_date',
    'end_date',
    'validity',
  ];

  /**
   * Gets the explanation message that was shown to the declarer.
   *
   * @return string
   *   Explanation message shown to declarer about their responsibilities.
   */
  public function getExplanation();

  /**
   * Sets the explanation message that was shown to the declarer.
   *
   * @param string $explanation
   *   Explanation message shown to declarer about their responsibilities.
   *
   * @return $this
   */
  public function setExplanation(string $explanation);

  /**
   * Calculates the cancellation status of the declaration.
   *
   * @return $this
   */
  public function calculateCancellation();

  /**
   * Gets the date that the declaration was cancelled.
   *
   * @param bool $formatted
   *   (optional) TRUE to return the date as an ISO format string.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|string|null
   *   Cancellation date as an object or string or null if the date isn't set.
   */
  public function getCancellationDate(bool $formatted = FALSE);

  /**
   * Whether the declaration has a cancellation date specified.
   *
   * @return bool
   *   Whether the declaration has a cancellation date.
   */
  public function hasCancellationDate();

  /**
   * Gets whether the declaration is currently cancelled.
   *
   * @return bool
   *   Whether the declaration is cancelled.
   */
  public function isCancelled();

  /**
   * Whether a written confirmation of this declaration has been sent to the declarer.
   *
   * @return bool
   *   Whether a written confirmation has been sent.
   */
  public function isConfirmationSent();

  /**
   * Gets the date a written confirmation was most recently sent.
   *
   * @param bool $formatted
   *   (optional) TRUE to return the date as an ISO format string.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|string|null
   *   Written confirmation date as an object or string or null if the date isn't set.
   */
  public function getConfirmationDate(bool $formatted = FALSE);

  /**
   * Sets the date a written confirmation was sent.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime|string|null $date
   *   The written confirmation date as an object or string (in ISO format) or NULL.
   *
   * @return $this
   */
  public function setConfirmationDate(string $date);

  /**
   * Gets the date when the declaration last had a significant change.
   *
   * @param bool $formatted
   *   (optional) TRUE to return the date as an ISO format string.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|string|null
   *   Changed date as an object or string or null if the date isn't set.
   */
  public function getChangedDate(bool $formatted = FALSE);

  /**
   * Sets the date when the declaration last had a significant change.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime|string|null $date
   *   The changed date as an object or string (in ISO format) or NULL.
   *
   * @return $this
   */
  public function setChangedDate($date);

  /**
   * Gets whether the declaration is locked, so key fields can't be changed.
   *
   * @return bool
   *   Whether the declaration is locked.
   */
  public function isLocked();

  /**
   * Gets whether the declaration can safely be used to make a claim.
   *
   * @return bool
   *   Whether the declaration is claimable.
   */
  public function isClaimable();

  /**
   * Whether the declaration is valid.
   *
   * Declarations are invalid if they lack sufficient auditable information.
   *
   * @return bool
   *   Whether the declaration is valid.
   */
  public function isValid();

  /**
   * Whether a written confirmation is required and missing for this declaration.
   *
   * @return bool
   *   Whether a written confirmation is missing.
   */
  public function isConfirmationMissing();

  /**
   * Gets the validity.
   *
   * @return string
   *   The validity: DECLARATION_INVALID, DECLARATION_VALID_IF_CONFIRMED or
   *   DECLARATION_INHERENTLY_VALID.
   */
  public function getValidity();

  /**
   * Sets the validity.
   *
   * @param string $valid
   *   The validity: DECLARATION_INVALID, DECLARATION_VALID_IF_CONFIRMED or
   *   DECLARATION_INHERENTLY_VALID.
   *
   * @return $this
   */
  public function setValidity(string $valid);

  /**
   * Gets the declaration status.
   *
   * @return string
   *   The status.
   */
  public function getStatus();

  /**
   * Gets a human-readable description status.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The status string.
   */
  public function getStatusString();

  /**
   * Gets whether the declaration is ongoing.
   *
   * A declaration is ongoing if it is valid, confirmed if necessary, claimable
   * now and not cancelled.
   *
   * @return bool
   *   Whether the declaration is ongoing.
   */
  public function isOngoing();

  /**
   * Gets the allowed options for the validity field.
   *
   * @return string[]
   *   Map of status value to status string.
   */
  public static function validityOptions();

  /**
   * Gets the allowed options for the status field.
   *
   * @return string[]
   *   Map of status value to status string.
   */
  public static function statusOptions();

}
