<?php

namespace Drupal\gift_aid\Entity;

use CommerceGuys\Addressing\AddressFormat\AddressField;
use CommerceGuys\Addressing\AddressFormat\FieldOverride;
use Drupal\address\AddressInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\gift_aid\EntityValidationTrait;

/**
 * Defines the Donor entity.
 *
 * @ingroup gift_aid
 *
 * @ContentEntityType(
 *   id = "gift_aid_donor",
 *   label = @Translation("Gift Aid donor"),
 *   label_collection = @Translation("Gift Aid donors"),
 *   label_singular = @Translation("Gift Aid donor"),
 *   label_plural = @Translation("Gift Aid donors"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Gift Aid donor",
 *     plural = "@count Gift Aid donors",
 *   ),
 *   handlers = {
 *     "view_builder" = "Drupal\gift_aid\Donor\DonorViewBuilder",
 *     "list_builder" = "Drupal\gift_aid\Donor\DonorListBuilder",
 *     "storage" = "Drupal\gift_aid\Donor\DonorStorage",
 *     "views_data" = "Drupal\gift_aid\Donor\DonorViewsData",
 *     "form" = {
 *       "default" = "Drupal\gift_aid\Form\Donor\DonorForm",
 *       "context" = "Drupal\gift_aid\Form\Donor\DonorContextForm",
 *       "delete" = "Drupal\gift_aid\Form\GiftAidDeleteForm",
 *       "cancel" = "Drupal\gift_aid\Form\Donor\DonorCancelForm",
 *     },
 *     "access" = "Drupal\gift_aid\Donor\DonorAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider",
 *       "revision" = "Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "gift_aid_donor",
 *   data_table = "gift_aid_donor_field_data",
 *   revision_table = "gift_aid_donor_revision",
 *   revision_data_table = "gift_aid_donor_field_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer gift aid declarations",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "uuid" = "uuid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log"
 *   },
 *   links = {
 *     "canonical" = "/admin/gift-aid/donor/{gift_aid_donor}",
 *     "add-form" = "/admin/gift-aid/donor/add",
 *     "edit-form" = "/admin/gift-aid/donor/{gift_aid_donor}/edit",
 *     "delete-form" = "/admin/gift-aid/donor/{gift_aid_donor}/delete",
 *     "collection" = "/admin/gift-aid/donor",
 *     "version-history" = "/admin/gift-aid/donor/{gift_aid_donor}/revisions",
 *     "revision" = "/admin/gift-aid/donor/{gift_aid_donor}/revisions/{gift_aid_donor_revision}/view",
 *   },
 *   field_ui_base_route = "entity.gift_aid_donor.collection",
 * )
 */
class Donor extends RevisionableContentEntityBase implements DonorInterface {

  use EntityValidationTrait;

  /**
   * {@inheritdoc}
   */
  public function getContext() {
    return $this->get('context')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setContext(EntityInterface $context) {
    $this->set('context', $context);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAddress() {
    return $this->get('address')->first();
  }

  /**
   * {@inheritdoc}
   */
  public function getAddressName() {
    $address = $this->getAddress();
    return $address->getGivenName() . ' ' . $address->getFamilyName();
  }

  /**
   * {@inheritdoc}
   */
  public function setAddress(AddressInterface $address) {
    return $this->set('address', $address->getValue());
  }

  /**
   * {@inheritdoc}
   */
  public function getDeclarations() {
    if ($this->id()) {
      return \Drupal::entityTypeManager()
        ->getStorage('gift_aid_declaration')
        ->loadByProperties(['donor' => $this->id()]);
    }
    else {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCancellations() {
    if ($this->id()) {
      return \Drupal::entityTypeManager()
        ->getStorage('gift_aid_cancellation')
        ->loadByProperties(['donor' => $this->id()]);
    }
    else {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus() {
    $declarations = $this->getDeclarations();
    $status = $declarations ? static::DONOR_DECL_NOT_CURRENT : static::DONOR_DECL_NONE;

    foreach ($declarations as $declaration) {
      if ($declaration->isOngoing()) {
        $status = static::DONOR_DECL_INCOMPLETE;
        if (!$declaration->hasEndDate() && !$declaration->hasCancellationDate()) {
          return static::DONOR_DECL_COMPLETE;
        }
      }
    }
    return $status;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusString() {
    return static::statusOptions()[$this->getStatus()];
  }

  /**
   * {@inheritdoc}
   */
  public function hasOngoing(?CharityInterface $charity = NULL) {
    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration');
    return $storage->hasOngoingByContext($this->getContext(), $charity);
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    if ($context = $this->getContext()) {
      return $context->label();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isCurrentUser(?AccountInterface $current = NULL) {
    $current ??= \Drupal::currentUser();
    return $current->isAuthenticated() && $this->getContext() && $this->getContext()->getEntityTypeId() === 'user' && $this->getContext()->id() === $current->id();
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['address'] = BaseFieldDefinition::create('address')
      ->setLabel(t('Address'))
      ->setDescription(t('Name and address for tax purposes.'))
      ->setSetting('field_overrides', [
        AddressField::ORGANIZATION => ['override' => FieldOverride::HIDDEN],
        AddressField::ADDRESS_LINE3 => ['override' => FieldOverride::HIDDEN],
      ])
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', ['weight' => 0])
      ->setDisplayOptions('form', ['weight' => 0])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['context'] = BaseFieldDefinition::create('dynamic_entity_reference')
      ->setLabel(t('Context'))
      ->setDescription(t('The context, used to link it to another entity.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', ['weight' => 0])
      ->setDisplayOptions('form', ['weight' => 0])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['evidence'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Evidence'))
      ->setDescription(t('The evidence for this change.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'gift_aid_evidence')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'type' => 'entity_reference_entity_view',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'inline_entity_form_complex',
        'weight' => 0,
        'settings' => [
          'removed_reference' => 'delete',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields += static::revisionLogBaseFieldDefinitions($entity_type);
    $fields['revision_log']->setLabel(t('Notes'));
    $fields['revision_log']->setDescription(t("Enter any notes about the donor."));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public static function statusOptions() {
    return [
      DonorInterface::DONOR_DECL_NONE => t("None"),
      DonorInterface::DONOR_DECL_NOT_CURRENT => t("Not currently valid"),
      DonorInterface::DONOR_DECL_INCOMPLETE => t("Incomplete"),
      DonorInterface::DONOR_DECL_COMPLETE => t("Complete"),
    ];
  }

}
