<?php

namespace Drupal\gift_aid\Entity;

use Drupal\address\AddressInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Provides an interface for defining donors.
 *
 * @ingroup gift_aid
 */
interface DonorInterface extends ContentEntityInterface, RevisionLogInterface {

  // Donor declaration status constants.
  const DONOR_DECL_NONE = 'none';
  const DONOR_DECL_NOT_CURRENT = 'not current';
  const DONOR_DECL_INCOMPLETE = 'incomplete';
  const DONOR_DECL_COMPLETE = 'complete';

  /**
   * Gets declaration context.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   An entity that is the context for the declaration.
   */
  public function getContext();

  /**
   * Set declaration context.
   *
   * @param \Drupal\Core\Entity\EntityInterface $context
   *   An entity that is the context for the declaration.
   *
   * @return $this
   */
  public function setContext(EntityInterface $context);

  /**
   * Gets the address.
   *
   * @return \Drupal\address\AddressInterface
   *   The address.
   */
  public function getAddress();

  /**
   * Gets a string containing the name part of the address.
   *
   * @return string
   *   The name from the address.
   */
  public function getAddressName();

  /**
   * Sets the address.
   *
   * @param \Drupal\address\AddressInterface $address
   *   The address.
   *
   * @return $this
   */
  public function setAddress(AddressInterface $address);

  /**
   * Gets the declarations associated with this donor.
   *
   * @return Drupal\gift_aid\Entity\DeclarationInterface[]
   *   The declarations.
   */
  public function getDeclarations();

  /**
   * Gets the cancellations associated with this donor.
   *
   * @return Drupal\gift_aid\Entity\CancellationInterface[]
   *   The cancellations.
   */
  public function getCancellations();

  /**
   * Gets a human-readable description of the declaration status of this donor.
   *
   * The result is based on the declaration with the "best" validity.
   *
   * @return string
   *   The status.
   */
  public function getStatus();

  /**
   * Gets a human-readable description of the declaration status.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The status string.
   */
  public function getStatusString();

  /**
   * Gets whether the Donor has an ongoing declaration.
   *
   * @param \Drupal\gift_aid\Entity\CharityInterface $charity
   *   (optional) The charity to find declarations for.
   *
   * @return bool
   *   Whether there is an ongoing declaration.
   */
  public function hasOngoing(CharityInterface $charity);

  /**
   * Determines if the donor is associated with the current user.
   *
   * @param Drupal\Core\Session\AccountInterface $current
   *   An account to check against, instead of the current user.
   *
   * @return bool
   *   TRUE if the donor is current, FALSE otherwise.
   */
  public function isCurrentUser(?AccountInterface $current = NULL);

  /**
   * Gets the allowed options for the status field.
   *
   * @return string[]
   *   Map of status value to status string.
   */
  public static function statusOptions();

}
