<?php

namespace Drupal\gift_aid\Entity;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItem;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\gift_aid\EntityValidationTrait;

/**
 * Defines the common base class for Declaration and Cancellation entities.
 */
abstract class RecordBase extends RevisionableContentEntityBase implements RecordInterface {

  use EntityValidationTrait {
    preSave as doValidate;
  }

  // Understand Gift Aid declaration dates to refer to days in this timezone.
  const GIFT_AID_TIMEZONE = 'Europe/London';

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    if (isset($values['declared_date']) && !isset($values['start_date'])) {
      $values['start_date'] = $values['declared_date'];
    }
    $today = static::today();
    $values += [
      'start_date' => $today,
      'declared_date' => $today,
      'charity' => \Drupal::entityTypeManager()->getStorage('gift_aid_charity')->loadDefault(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function preDelete(EntityStorageInterface $storage, array $entities) {
    parent::preDelete($storage, $entities);

    /** @var \Drupal\gift_aid\Entity\Declaration $declaration */
    foreach ($entities as $declaration) {
      if ($evidence = $declaration->getEvidence()) {
        $evidence->delete();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    if (!$this->isDateBased()) {
      $this->setStartDate(NULL);
      $this->setEndDate(NULL);
    }

    parent::preSave($storage);
    $this->doValidate($storage);
  }

  /**
   * {@inheritdoc}
   */
  public function getDonor() {
    return $this->get('donor')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setDonor(DonorInterface $donor) {
    $this->set('donor', $donor->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDonorId() {
    return $this->get('donor')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setDonorId(int $donor_id) {
    $this->set('donor', $donor_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCharity() {
    return $this->get('charity')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setCharity(CharityInterface $charity) {
    $this->set('charity', $charity->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCharityId() {
    return $this->get('charity')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setCharityId(int $charity_id) {
    $this->set('charity', $charity_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getEvidence() {
    return $this->get('evidence')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setEvidence(EvidenceInterface $evidence) {
    $this->set('evidence', $evidence->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDeclaredDate(bool $formatted = FALSE) {
    return $this->getDateField('declared_date', $formatted);
  }

  /**
   * {@inheritdoc}
   */
  public function setDeclaredDate($date) {
    return $this->setDateField('declared_date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function hasStarted() {
    $start = $this->getStartDate(TRUE);
    if (!$this->isDateBased() || !$start) {
      return FALSE;
    }
    return ($start <= static::today());
  }

  /**
   * {@inheritdoc}
   */
  public function getStartDate(bool $formatted = FALSE) {
    return $this->getDateField('start_date', $formatted, [0, 0, 0]);
  }

  /**
   * {@inheritdoc}
   */
  public function setStartDate($date) {
    return $this->setDateField('start_date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function hasEnded() {
    $end = $this->getEndDate(TRUE);
    if (!$this->isDateBased() || !$end) {
      return FALSE;
    }
    return ($end < static::today());
  }

  /**
   * {@inheritdoc}
   */
  public function getEndDate(bool $formatted = FALSE) {
    return $this->getDateField('end_date', $formatted, [23, 59, 59]);
  }

  /**
   * {@inheritdoc}
   */
  public function hasEndDate() {
    return !$this->get('end_date')->isEmpty();
  }

  /**
   * {@inheritdoc}
   */
  public function setEndDate($date) {
    return $this->setDateField('end_date', $date);
  }

  /**
   * {@inheritdoc}
   */
  public function isDateBased() {
    return !$this->get('date_based')->isEmpty() && $this->get('date_based')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setDateBased($status) {
    $this->set('date_based', (bool) $status);
    return $this;
  }

  /**
   * Gets a date field.
   *
   * @param string $fieldName
   *   Name of the field to get.
   * @param bool $formatted
   *   TRUE to return the date as an ISO format string.
   * @param array $time
   *   (optional) Time to set as an array of hour, minute, second.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|string|null
   *   Date as an object or string or null if the date isn't set.
   */
  protected function getDateField(string $fieldName, bool $formatted, array $time = []) {
    $value = $this->get($fieldName)->value;
    if ($formatted || is_null($value)) {
      return $value;
    }
    $date = DrupalDateTime::createFromFormat(DateTimeItemInterface::DATE_STORAGE_FORMAT, $value, static::GIFT_AID_TIMEZONE);
    if ($time) {
      $date->setTime(...$time);
    }
    return $date;
  }

  /**
   * Sets a date field.
   *
   * @param string $fieldName
   *   Name of the field to set.
   * @param \Drupal\Core\Datetime\DrupalDateTime|string|null $date
   *   The date as an object or string (in ISO format) or NULL.
   *
   * @return $this
   */
  protected function setDateField(string $fieldName, $date) {
    if ($date instanceof DrupalDateTime) {
      $date->setTimezone(new \DateTimeZone(static::GIFT_AID_TIMEZONE));
      $date = $date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
    }
    $this->set($fieldName, $date);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['donor'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Donor'))
      ->setSetting('target_type', 'gift_aid_donor')
      ->setDescription(t('The donor for the declaration.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', ['weight' => 0])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['charity'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Charity'))
      ->setDescription(t('The charity the declaration is for.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'gift_aid_charity')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['evidence'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Additional evidence'))
      ->setDescription(t('Additional evidence for the declaration.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'gift_aid_evidence')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'type' => 'entity_reference_entity_view',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'inline_entity_form_complex',
        'weight' => 0,
        'settings' => [
          'removed_reference' => 'delete',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['declared_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Declared date'))
      ->setDescription(t('The date the declaration was made by the donor.'))
      ->setRevisionable(TRUE)
      ->setSetting('datetime_type', DateTimeItem::DATETIME_TYPE_DATE)
      ->setDisplayOptions('view', [
        'type' => 'datetime_default',
        'label' => 'inline',
        'weight' => 0,
        'settings' => [
          'format_type' => 'html_date',
        ],
      ])
      ->setDisplayOptions('form', ['weight' => 0])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['date_based'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Date-based'))
      ->setDescription(t('Is this a date-based declaration that covers all donations within a period of time?'))
      ->setDefaultValue(TRUE)
      ->setRevisionable(TRUE)
      ->setSettings([
        'on_label' => t('Yes'),
        'off_label' => t('No'),
      ])
      ->setDisplayOptions('view', [
        // Hidden by default as date-based is the only supported approach without custom code.
        'region' => 'hidden',
        'label' => 'inline',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        // Hidden by default as date-based is the only supported approach without custom code.
        'region' => 'hidden',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['start_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Start date'))
      ->setDescription(t('Only donations made after this date are covered by the declaration.'))
      ->setRevisionable(TRUE)
      ->setSetting('datetime_type', DateTimeItem::DATETIME_TYPE_DATE)
      ->setDisplayOptions('view', [
        'type' => 'datetime_default',
        'label' => 'inline',
        'weight' => 0,
        'settings' => [
          'format_type' => 'html_date',
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_default',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['end_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('End date'))
      ->setDescription(t('Donations made after this date are not covered by the declaration.'))
      ->setRevisionable(TRUE)
      ->setSetting('datetime_type', DateTimeItem::DATETIME_TYPE_DATE)
      ->setDisplayOptions('view', [
        'type' => 'datetime_default',
        'label' => 'inline',
        'weight' => 0,
        'settings' => [
          'format_type' => 'html_date',
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_default',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields += static::revisionLogBaseFieldDefinitions($entity_type);
    $fields['revision_log']->setLabel(t('Notes'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public static function today(string $date = 'now') {
    $date = new DrupalDateTime($date, static::GIFT_AID_TIMEZONE);
    return $date->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
  }

}
