<?php

namespace Drupal\gift_aid\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\RevisionLogInterface;

/**
 * Defines the common base interface for Declaration and Cancellation entities.
 */
interface RecordInterface extends ContentEntityInterface, RevisionLogInterface {

  /**
   * Gets the Donor.
   *
   * @return \Drupal\gift_aid\Entity\DonorInterface
   *   The Donor entity.
   */
  public function getDonor();

  /**
   * Sets the Donor entity.
   *
   * @param \Drupal\gift_aid\Entity\DonorInterface $donor
   *   Donor entity.
   *
   * @return $this
   */
  public function setDonor(DonorInterface $donor);

  /**
   * Gets the Donor ID.
   *
   * @return int
   *   The Donor entity ID.
   */
  public function getDonorId();

  /**
   * Sets the Donor entity ID.
   *
   * @param int $donor_id
   *   Donor entity ID.
   *
   * @return $this
   */
  public function setDonorId(int $donor_id);

  /**
   * Gets the Charity.
   *
   * @return \Drupal\gift_aid\Entity\CharityInterface
   *   The Charity entity.
   */
  public function getCharity();

  /**
   * Sets the Charity entity.
   *
   * @param \Drupal\gift_aid\Entity\CharityInterface $charity
   *   Charity entity.
   *
   * @return $this
   */
  public function setCharity(CharityInterface $charity);

  /**
   * Gets the Charity ID.
   *
   * @return int
   *   The Charity entity ID.
   */
  public function getCharityId();

  /**
   * Sets the Charity entity ID.
   *
   * @param int $charity_id
   *   Charity entity ID.
   *
   * @return $this
   */
  public function setCharityId(int $charity_id);

  /**
   * Gets the Evidence.
   *
   * @return \Drupal\gift_aid\Entity\EvidenceInterface
   *   The Evidence entity.
   */
  public function getEvidence();

  /**
   * Sets the Evidence.
   *
   * @param \Drupal\gift_aid\Entity\EvidenceInterface $evidence
   *   Evidence entity.
   *
   * @return $this
   */
  public function setEvidence(EvidenceInterface $evidence);

  /**
   * Gets the date the declaration was declared.
   *
   * @param bool $formatted
   *   (optional) TRUE to return the date as an ISO format string.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|string
   *   Declared date as an object or formatted string.
   */
  public function getDeclaredDate(bool $formatted = FALSE);

  /**
   * Sets the declared date.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime|string $date
   *   The declared date as an object or string (in ISO format).
   *
   * @return $this
   */
  public function setDeclaredDate($date);

  /**
   * Whether or not a date-based declaration has taken effect.
   *
   * @return bool
   *   Whether the start date has passed.
   */
  public function hasStarted();

  /**
   * Gets the start date of the period covered by this declaration.
   *
   * @param bool $formatted
   *   (optional) TRUE to return the date as an ISO format string.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|string|null
   *   Start date as an object or string or NULL if the start date isn't set.
   */
  public function getStartDate(bool $formatted = FALSE);

  /**
   * Sets the start date.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime|string|null $date
   *   The start date as an object or string (in ISO format) or NULL.
   *
   * @return $this
   */
  public function setStartDate($date);

  /**
   * Whether or not the gift aid declaration has ended.
   *
   * @return bool
   *   Whether the end date has passed.
   */
  public function hasEnded();

  /**
   * Gets the end date of the period covered by this declaration.
   *
   * @param bool $formatted
   *   (optional) TRUE to return the date as an ISO format string.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|string|null
   *   End date as an object or string or null if the end date isn't set.
   */
  public function getEndDate(bool $formatted = FALSE);

  /**
   * Whether the declaration has an end date specified.
   *
   * @return bool
   *   Whether the declaration has an end date.
   */
  public function hasEndDate();

  /**
   * Sets the end date.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime|string|null $date
   *   The end date as an object or string (in ISO format) or NULL.
   *
   * @return $this
   */
  public function setEndDate($date);

  /**
   * Whether or not the declaration is date-based, applying to any donation between the start and end dates.
   *
   * @return bool
   *   Whether the declaration is date-based.
   */
  public function isDateBased();

  /**
   * Set whether the declaration is date-based.
   *
   * @param bool $date_based
   *   Whether the declaration is date_based.
   *
   * @return $this
   */
  public function setDateBased($date_based);

  /**
   * Gets a date string for today or the specified date.
   *
   * @param string $date
   *   The date to fetch.
   *
   * @return string
   *   Date string (in ISO format) for the specified date.
   */
  public static function today(string $date = 'now');

}
