<?php

namespace Drupal\gift_aid\Event;

use Drupal\Component\EventDispatcher\Event;
use Drupal\Core\Entity\EntityInterface;

/**
 * Provides a event that allows listeners to add additional related contexts.
 */
class GiftAidRelatedContextsEvent extends Event {

  /**
   * The contexts.
   *
   * An array keyed by entity type, where the values are arrays with keys and values that are entity ids of that type.
   *
   * @var array
   */
  protected $contexts;

  /**
   * The added contexts not supplied in the constructor.
   *
   * An array keyed by entity type, where the values are arrays with keys and values that are entity ids of that type.
   *
   * @var array
   */
  protected $addedContexts = [];

  /**
   * The cache tags to use to invalidate the collection of discovered contexts.
   *
   * @var array
   */
  protected $cacheTags = [];

  /**
   * The original context, that all other contexts are related to.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected $original;

  /**
   * Constructs a gift aid related contexts event object.
   *
   * @param \Drupal\Core\Entity\EntityInterface $original
   *   The original context.
   * @param \Drupal\Core\Entity\EntityInterface[] $contexts
   *   Contexts already known to be related to the original context.
   * @param string[] $cache_tags
   *   Cache tags already known to invalidate the already discovered contexts.
   */
  public function __construct(EntityInterface $original, $contexts, $cache_tags) {
    $this->original = $original;
    $this->contexts = $contexts;
    $this->contexts[$original->getEntityTypeId()][$original->id()] = $original->id();
    $this->cacheTags = $cache_tags;
  }

  /**
   * Gets the original context.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The original context.
   */
  public function getOriginalContext() {
    return $this->original;
  }

  /**
   * Gets the related contexts, including the original.
   *
   * @return array
   *   The contexts array, keyed by entity type, where the values are arrays with entity ids as keys and values.
   */
  public function getContexts() {
    return $this->contexts;
  }

  /**
   * Gets the added contexts, those not supplied in the constructor.
   *
   * @return array
   *   The contexts array, keyed by entity type, where the values are arrays with entity ids as keys and values.
   */
  public function getAddedContexts() {
    return $this->addedContexts;
  }

  /**
   * Gets the count of contexts.
   *
   * @return int
   *   How many contexts there are across all entity types, including the original context.
   */
  public function getContextsCount() {
    return array_sum(array_map("count", $this->contexts));
  }

  /**
   * Adds a related context.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   *   The related context to add.
   */
  public function addContext(EntityInterface $context) {
    $this->addContextIds($context->getEntityTypeId(), [$context->id()]);
  }

  /**
   * Adds related contexts as entity ids.
   *
   * @var string $entity_type
   *   The entity type id.
   * @var array $entity_ids
   *   The entity ids.
   */
  public function addContextIds($entity_type, array $entity_ids) {
    $entity_ids = array_combine($entity_ids, $entity_ids);
    $merged_ids = ($this->contexts[$entity_type] ?? []) + $entity_ids;
    $this->contexts[$entity_type] = $merged_ids;

    $merged_ids = ($this->addedContexts[$entity_type] ?? []) + $entity_ids;
    $this->addedContexts[$entity_type] = $merged_ids;
  }

  /**
   * Adds a cache tag.
   *
   * @var array
   *   The cache tags to add.
   */
  public function addCacheTags($tags) {
    $this->cacheTags = array_unique(array_merge($this->cacheTags, $tags));
  }

  /**
   * Gets the cache tags that invalidate the discovered contexts.
   *
   * @var array
   *   an array of cache tags.
   */
  public function getCacheTags() {
    return $this->cacheTags;
  }

}
