<?php

namespace Drupal\gift_aid\Form\Donor;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\gift_aid\Entity\CharityInterface;
use Drupal\gift_aid\Entity\DeclarationInterface;

/**
 * Form controller for Donor edit in a context, for example as a tab on a user.
 *
 * @ingroup gift_aid
 */
class DonorContextForm extends DonorForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    /** @var \Drupal\gift_aid\Entity\DonorInterface $donor */
    $donor = $this->entity;

    if (!$donor->isCurrentUser()) {
      $form['status'] = [
        '#markup' => $this->t("Please enter the donor's address."),
        '#weight' => -1,
      ];
    }
    else {
      $this->addGiftAidFields($form);
      $form['revision_information']['#access'] = FALSE;
    }

    return $form;
  }

  /**
   * Adds Gift Aid fields to the specified form.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param ?\Drupal\gift_aid\Entity\CharityInterface $charity
   *   The charity to use, or NULL to use the default.
   *
   * @internal
   */
  public function addGiftAidFields(array &$form, ?CharityInterface $charity = NULL) {
    /** @var \Drupal\gift_aid\Entity\DonorInterface $donor */
    $donor = $this->entity;
    $charity ??= $this->entityTypeManager->getStorage('gift_aid_charity')->loadDefault();

    $ongoing = $donor->hasOngoing();
    $status_text = $ongoing ?
      $this->t("Thank you for making a Gift Aid declaration so that we can claim tax back on your declarations.") :
      $this->t("Use this page to make a Gift Aid declaration so that we can claim tax back on your declarations.");

    $form['above'] = [
      '#type' => 'container',
      '#weight' => -1,
      '#attributes' => ['class' => ["gift-aid-context__above"]],
      'logo' => [
        '#type' => 'container',
        '#attributes' => ['class' => ["gift-aid-context__logo"]],
      ],
      'header' => [
        '#type' => 'html_tag',
        '#tag' => 'h3',
        '#value' => $this->t("Increase your donation at no extra cost to you"),
      ],
      'status' => ['#markup' => $status_text],
    ];

    $form['below'] = [
      '#type' => 'container',
      '#weight' => 90,
      'call_to_action' => ['#markup' => $this->t("I am a UK tax payer and would like Gift Aid claimed on my donations.")],
      'explanation' => [
        '#type' => 'container',
        'contents' => [
          '#type' => 'inline_template',
          '#template' => $charity->getExplanation(),
          '#context' => ['charity' => $charity],
        ],
        '#attributes' => ['class' => ["gift-aid-context__explanation"]],
      ],
      '#access' => !$ongoing,
    ];

    $form['#attached']['library'][] = 'gift_aid/context';
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\gift_aid\Entity\DonorInterface $donor */
    $donor = $this->entity;
    $self = $donor->isCurrentUser();
    $status = ContentEntityForm::save($form, $form_state);

    if ($self) {
      // A self-declaration can be created immediately.
      /** @var \Drupal\gift_aid\Entity\DeclarationInterface $declaration */
      $declaration = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration')->create(['type' => 'web']);
      $declaration
        ->setDonor($donor)
        ->setExplanation($form['below']['explanation']['contents']['#template'])
        ->setValidity(DeclarationInterface::DECLARATION_INHERENTLY_VALID)
        ->save();

      $this->messenger()->addStatus($this->t('Thank you for using Gift Aid.'));
    }
    else {
      // If not a self-declaration, redirect to the declaration add form to add evidence.
      $form_state->setRedirect('entity.gift_aid_declaration.add_page', [], ['query' => ['donor' => $donor->id()]]);
    }

    return $status;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    /** @var \Drupal\gift_aid\Entity\DonorInterface $donor */
    $donor = $this->entity;
    if (!$donor->isCurrentUser()) {
      $actions['submit']['#value'] = $this->t('Next');
    }
    elseif (!$donor->hasOngoing()) {
      $actions['submit']['#value'] = $this->t('Agree');
    }
    else {
      $actions['submit']['#value'] = $this->t('Change address');

      $route_match = $this->getRouteMatch();
      $url = Url::fromRoute($route_match->getRouteName() . '.cancel', $route_match->getRawParameters()->all());
      $actions['cancel'] = $actions['delete'];
      $actions['cancel']['#title'] = $this->t('Cancel Gift Aid');
      $actions['cancel']['#access'] = $url->access();
      $actions['cancel']['#url'] = $url;
    }

    unset($actions['delete']);
    return($actions);
  }

}
