<?php

namespace Drupal\gift_aid\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for Declaration or Cancellation edit forms.
 *
 * @ingroup gift_aid
 */
class RecordForm extends ContentEntityForm {

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->messenger = $container->get('messenger');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareEntity() {
    parent::prepareEntity();

    // Allow donor and charity to be prepopulated by querystring when creating.
    if ($this->entity->isNew()) {
      $donor_id = \Drupal::request()->query->get('donor');
      if ($donor_id) {
        $this->entity->setDonorId($donor_id);
      }

      $charity_id = \Drupal::request()->query->get('charity');
      if ($charity_id) {
        $this->entity->setCharityId($charity_id);
      }

    }
  }

  /* @todo VALIDATION: only admins should be able to backdate end date,
   * except when creating.
   * Button text should vary depending on whether declaration is ongoing.
   */

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    // It's mandatory to set a new revision for compliance.
    $form['revision']['#disabled'] = TRUE;

    // Hide context and charity if set by querystring, but only when creating.
    if ($this->entity->isNew()) {
      if ($this->entity->getEntityTypeId() == 'gift_aid_declaration') {
        $bundle_label = $this->entity->type->entity->label();
        $intro[] = $this->t("Record a '@bundle_label' Gift Aid declaration.", ['@bundle_label' => $bundle_label]);
      }

      if (\Drupal::request()->query->get('donor')) {
        $form['donor']['#access'] = FALSE;
        $intro[] = $this->t("Donor: @donor", ['@donor' => $this->entity->getDonor()->label()]);
      }
      if (\Drupal::request()->query->get('charity')) {
        $form['charity']['#access'] = FALSE;
        $intro[] = $this->t('Charity: @charity', ['@charity' => $this->entity->getCharity()->label()]);
      }

      if (isset($intro)) {
        $form['intro'] = [
          '#type' => 'fieldset',
          '#title' => $this->t('Information'),
          'text' => [
            '#type' => 'markup',
            '#markup' => implode('<br>', $intro),
          ],
        ];
      }
    }

    // Hide charity if there is only one and it's the default.
    if (!isset($form['charity']['#access']) || $form['charity']['#access'] !== FALSE) {
      /** @var \Drupal\gift_aid\Entity\CharityStorageInterface $charity_storage */
      $charity_storage = $this->entityTypeManager->getStorage('gift_aid_charity');
      $charity_ids = $charity_storage->getQuery()->accessCheck(FALSE)->execute();
      if (count($charity_ids) === 1 && $charity_storage->loadDefault()) {
        $form['charity']['#access'] = FALSE;
      }
    }

    // @todo Show/hide Start and End dates depending on the DateBased state.
    /*    $date_based_checked = [
    ':input[name="date-based[value]"]' => ['checked' => TRUE],
    ];
    $form['start_date']['#states'] = [
    'visible' => $date_based_checked,
    ];
    $form['start_date']['widget'][0]['value']['#states'] = [
    'required' => $date_based_checked,
    ];
    $form['end_date']['#states'] = [
    'visible' => $date_based_checked,
    ];*/

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\gift_aid\Entity\RecordInterface $entity */
    $entity = $this->entity;
    $status = parent::save($form, $form_state);
    $params = ['%label' => $entity->getDonor()->label()];

    if ($entity->getEntityTypeId() == 'gift_aid_declaration') {
      $params['@bundle'] = $entity->type->entity->label();
      $message = ($status == SAVED_NEW) ?
        $this->t('Recorded a @bundle Gift Aid declaration for %label.', $params) :
        $this->t('Updated the @bundle Gift Aid declaration for %label.', $params);
    }
    else {
      $message = ($status == SAVED_NEW) ?
        $this->t('Recorded a Gift Aid cancellation for %label.', $params) :
        $this->t('Updated the Gift Aid cancellation for %label.', $params);
    }

    $this->messenger->addMessage($message);
    $form_state->setRedirectUrl($entity->getDonor()->toUrl());
    return $status;
  }

  /**
   * {@inheritdoc}
   */
  protected function getNewRevisionDefault() {
    return TRUE;
  }

}
