<?php

namespace Drupal\gift_aid;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\MemoryCache\MemoryCacheInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\gift_aid\Event\GiftAidEvents;
use Drupal\gift_aid\Event\GiftAidRelatedContextsEvent;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Finds related gift aid declaration contexts.
 */
class GiftAidRelatedContextsManager implements GiftAidRelatedContextsManagerInterface {

  /**
   * The event in progress.
   *
   * @var \Drupal\gift_aid\Event\GiftAidRelatedContextsEvent
   */
  protected $event;

  /**
   * Constructs a new GiftAidRelatedContextsManager.
   *
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Cache\MemoryCache\MemoryCacheInterface $cache
   *   The cache.
   */
  public function __construct(protected readonly EventDispatcherInterface $eventDispatcher, protected readonly MemoryCacheInterface $cache) {
  }

  /**
   * {@inheritdoc}
   */
  public function getRelatedContexts(EntityInterface $original_context) {
    $cache_id = $this->getCacheId($original_context->getEntityTypeId(), $original_context->id());
    $cache = $this->cache->get($cache_id);

    if ($cache) {
      $result = $cache->data;
    }
    else {
      $this->discoverRelatedContexts($original_context);
      $result = $this->event->getContexts();
      $this->cache->set($cache_id, $result, Cache::PERMANENT, $this->event->getCacheTags());
    }
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function isRelatedContext(EntityInterface $original_context, EntityInterface $other_context) {
    $contexts = $this->getRelatedContexts($original_context);
    return isset($contexts[$other_context->getEntityTypeId()][$other_context->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags($original_context) {
    // Ensure context is cached.
    $this->getRelatedContexts($original_context);
    $cache_id = $this->getCacheId($original_context->getEntityTypeId(), $original_context->id());
    return $this->cache->get($cache_id)->tags;
  }

  /**
   * Build a cache id for a context.
   *
   * @param string $context_type
   *   The entity type id of the context.
   * @param string $context_id
   *   The id of the context entity.
   *
   * @return string
   *   The cache id.
   */
  protected function getCacheId(string $context_type, string $context_id) {
    return 'gift_aid:related_contexts:' . $context_type . ':' . $context_id;
  }

  /**
   * Discover additional contexts related to an original context.
   *
   * @param \Drupal\Core\Entity\EntityInterface $original_context
   *   The context to discover related contexts for.
   */
  protected function discoverRelatedContexts(EntityInterface $original_context) {
    $previous_count = -1;
    $event = new GiftAidRelatedContextsEvent($original_context, [], []);

    // Keep discovering contexts recursively until no more are found.
    while (array_sum(array_map('count', $event->getContexts())) > $previous_count) {
      // Benchmark how many contexts are currently known before looking for more.
      $previous_count = array_sum(array_map('count', $event->getContexts()));

      // Discover more contexts.
      $event = new GiftAidRelatedContextsEvent($original_context, $event->getContexts(), $event->getCacheTags());
      $this->eventDispatcher->dispatch($event, GiftAidEvents::RELATED_CONTEXTS);

      // Add in any cached contexts known from previously added contexts.
      $this->addCachedContexts($event);
    }
    $this->event = $event;
  }

  /**
   * Add cached contexts with tags from an event.
   *
   * @param \Drupal\gift_aid\Event\GiftAidRelatedContextsEvent $event
   *   The event to add contexts from.
   */
  protected function addCachedContexts(GiftAidRelatedContextsEvent $event) {
    foreach ($event->getAddedContexts() as $new_context_type => $new_context_ids) {
      foreach ($new_context_ids as $new_context_id) {
        $cache_id = $this->getCacheId($new_context_type, $new_context_id);
        $cached = $this->cache->get($cache_id);
        if ($cached) {
          foreach ($cached->data as $type => $ids) {
            $event->addContextIds($type, $ids);
          }
          $event->addCacheTags($cached->tags);
        }
      }
    }
  }

}
