<?php

namespace Drupal\gift_aid;

use Drupal\Core\Entity\EntityInterface;

/**
 * An interface for related gift aid declaration contexts.
 */
interface GiftAidRelatedContextsManagerInterface {

  /**
   * Get all related contexts from an original context.
   *
   * A context is related if the GiftAidEvents::RELATED_CONTEXTS event discovers it as related.
   *
   * @param \Drupal\Core\Entity\EntityInterface $original_context
   *   The entity that is the original context to find related contexts and declarations for.
   *
   * @return array
   *   An array keyed by entity type, where the values are arrays of entity ids for that entity type.
   */
  public function getRelatedContexts(EntityInterface $original_context);

  /**
   * Get the cache tags for the related contexts.
   *
   * If any of these cache tags are invalidated then the related contexts need recomputing.
   *
   * @param \Drupal\Core\Entity\EntityInterface $original_context
   *   The entity that is the original context to find related contexts and declarations for.
   *
   * @return array
   *   An array of cache tags.
   */
  public function getCacheTags(EntityInterface $original_context);

  /**
   * Establish whether one context is related to another.
   *
   * @param \Drupal\Core\Entity\EntityInterface $original_context
   *   The entity that is the original context to find related contexts and declarations for.
   * @param \Drupal\Core\Entity\EntityInterface $other_context
   *   The entity that may or may not be a related context to $original_context.
   *
   * @return bool
   *   Whether the two contexts are related.
   */
  public function isRelatedContext(EntityInterface $original_context, EntityInterface $other_context);

}
