<?php

namespace Drupal\gift_aid\Plugin\Field\FieldWidget;

use Drupal\Core\Form\FormStateInterface;
use Drupal\dynamic_entity_reference\Plugin\Field\FieldType\DynamicEntityReferenceItem;
use Drupal\dynamic_entity_reference\Plugin\Field\FieldWidget\DynamicEntityReferenceWidget;

/**
 * Implementation of the 'gift_aid_single_type_dynamic_entity_reference' widget.
 *
 * This simply extends the settings of the DER widget to allow specifying a
 * single entity type as available. This provides a superior UX in situations
 * where a site-builder knows that on their site only a single entity type is
 * used for a given field or form.
 *
 * @FieldWidget(
 *   id = "gift_aid_single_type_dynamic_entity_reference",
 *   label = @Translation("Single type autocomplete"),
 *   description = @Translation("An autocomplete text field limited to a single entity type."),
 *   field_types = {
 *     "dynamic_entity_reference"
 *   }
 * )
 */
class SingleTypeDynamicEntityReferenceWidget extends DynamicEntityReferenceWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'single_entity_type_id' => NULL,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  protected function getFieldSettings() {
    $settings = $this->fieldDefinition->getSettings();

    // Override the 'entity_type_ids' setting from the field.
    if (!empty($this->getSetting('single_entity_type_id'))) {
      $settings['entity_type_ids'] = [$this->getSetting('single_entity_type_id')];
      $settings['exclude_entity_types'] = FALSE;
    }

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    // Restrict options to those specified as available in the field settings.
    $labels = \Drupal::service('entity_type.repository')->getEntityTypeLabels(TRUE);
    $options = $labels[(string) $this->t('Content', [], ['context' => 'Entity type group'])];
    $available = DynamicEntityReferenceItem::getTargetTypes(parent::getFieldSettings());
    $available = array_combine($available, $available);
    $options = array_intersect_key($options, $available);

    $element['single_entity_type_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity type'),
      '#default_value' => $this->getSetting('single_entity_type_id') ?? reset($options),
      '#options' => $options,
      '#description' => $this->t('Only entities of this type can be referenced.'),
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    $labels = \Drupal::service('entity_type.repository')->getEntityTypeLabels(TRUE);
    $options = $labels[(string) $this->t('Content', [], ['context' => 'Entity type group'])];
    $entity_type = $this->getSetting('single_entity_type_id') ?? reset($options);
    $summary[] = $this->t('Entity type: @entity_type', ['@entity_type' => $entity_type]);

    return $summary;
  }

}
