<?php

namespace Drupal\gift_aid\Record;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Link;

/**
 * Defines a class to build a listing of declarations or cancellations.
 *
 * @ingroup gift_aid
 */
class RecordListBuilder extends EntityListBuilder {

  /**
   * The entities to include in the listing.
   */
  protected ?array $entities = NULL;

  /**
   * Sets the entities to include in the listing.
   *
   * @param array $entities
   *   The entities.
   */
  public function setEntities(?array $entities) {
    $this->entities = $entities;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['id'] = $this->t('ID');
    if ($decl = $this->entityTypeId == 'gift_aid_declaration') {
      $header['declaration_type'] = $this->t('Declaration Type');
    }
    $header['charity'] = $this->t('Charity');
    if (!$this->entities) {
      $header['donor'] = $this->t('Donor');
    }
    if ($decl) {
      $header['status'] = $this->t('Status');
    }
    $header['declared_date'] = $this->t('Declared Date');
    $header['start_date'] = $this->t('Start Date');
    $header['end_date'] = $this->t('End Date');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\gift_aid\Entity\RecordInterface $entity */
    $row['id'] = Link::fromTextAndUrl($entity->id(), $entity->toUrl());
    if ($decl = $this->entityTypeId == 'gift_aid_declaration') {
      $row['declaration_type'] = $entity->type->entity->toLink(NULL, 'edit-form')->toString();
    }
    $row['charity'] = $entity->getCharity()->toLink(NULL, 'edit-form')->toString();
    if (!$this->entities) {
      $row['donor'] = $entity->getDonor()->toLink()->toString();
    }
    if ($decl) {
      $row['status'] = $entity->getStatusString();
    }
    $row['declared_date'] = $entity->getDeclaredDate(TRUE);
    $row['start_date'] = $entity->getStartDate(TRUE) ?? '-';
    $row['end_date'] = $entity->getEndDate(TRUE) ?? '-';

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function load() {
    return $this->entities ?? parent::load();
  }

}
