<?php

namespace Drupal\Tests\gift_aid\Functional;

use Drupal\Core\Url;
use Drupal\gift_aid\Entity\DeclarationInterface;
use Drupal\gift_aid\Entity\DonorInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\gift_aid\Traits\GiftAidTestTrait;

/**
 * Tests the Gift Aid declaration create and edit forms.
 *
 * @group gift_aid
 */
class DeclarationFormTest extends BrowserTestBase {

  use GiftAidTestTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'gift_aid',
    'gift_aid_user',
    'gift_aid_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->charity = $this->createCharity(['is_default' => TRUE]);
  }

  /**
   * Tests a logged in user making a web declaration.
   */
  public function testWebDeclarationMadeByDonor() {
    // A donor makes a declaration.
    $donor_user = $this->createUser(['make gift aid declaration', 'cancel gift aid declaration']);
    $this->drupalLogin($donor_user);
    $url = Url::fromRoute('gift_aid_user.context', ['user' => $donor_user->id()]);
    $this->drupalGet($url->toString());
    $this->assertSession()->pageTextContains('Increase your donation at no extra cost to you');
    $this->assertSession()->pageTextContains('Use this page to make a Gift Aid declaration');
    $this->assertSession()->pageTextContains('I am a UK taxpayer');
    $this->assertSession()->pageTextContains('it is my responsibility to pay any difference');
    $this->assertSession()->pageTextNotContains('Thank you for making a Gift Aid declaration');
    $edit = [
      'address[0][address][given_name]' => $this->randomString(),
      'address[0][address][family_name]' => $this->randomString(),
      'address[0][address][address_line1]' => $this->randomString(),
      'address[0][address][locality]' => $this->randomString(),
    ];
    $this->submitForm($edit, 'Agree');
    $this->assertSession()->pageTextContains('Increase your donation at no extra cost to you');
    $this->assertSession()->pageTextContains('Thank you for making a Gift Aid declaration');
    $this->assertSession()->pageTextNotContains('Use this page to make a Gift Aid declaration');
    $this->assertSession()->pageTextNotContains('I am a UK taxpayer');

    // There should now be a stored donor and declaration.
    $donor = \Drupal::entityTypeManager()->getStorage('gift_aid_donor')->loadOrCreateByContext($donor_user);
    $this->assertFalse($donor->isNew());
    $this->assertEquals('user', $donor->getContext()->getEntityTypeId());
    $declarations = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration')->loadByContext($donor_user);
    $declaration = reset($declarations);
    $this->assertInstanceOf(DeclarationInterface::class, $declaration);
    $this->assertEquals('web', $declaration->getType());
    $this->assertEquals($this->charity->id(), $declaration->getCharity()->id());
    $this->assertTrue($declaration->isDateBased());
    $this->assertFalse($declaration->hasEndDate());
    $this->assertTrue($declaration->isOngoing());
    $this->assertFalse($declaration->isClaimable());
    $this->assertEquals($donor->id(), $declaration->getDonorId());

    // The donor can cancel the declaration.
    $this->clickLink('Cancel');
    $this->assertSession()->pageTextContains('Cancel Gift Aid');
    $this->submitForm([], 'Confirm');
    $this->assertSession()->pageTextContains('I am a UK taxpayer');
    $this->assertSession()->pageTextNotContains('Thank you for making a Gift Aid declaration');

    $declaration = $this->reloadDeclaration($declaration);
    $this->assertTrue($declaration->isCancelled());
    $this->assertEquals($declaration::today(), $declaration->getCancellationDate(TRUE));
    $this->assertEquals(DeclarationInterface::DECLARATION_CANCELLED, $declaration->getStatus());
    $this->assertEquals(DonorInterface::DONOR_DECL_NOT_CURRENT, $donor->getStatus());
    $this->assertFalse($declaration->isOngoing());
  }

  /**
   * Tests staff recording a declaration for someone else.
   */
  public function testDeclarationRecordedByStaff() {
    // Staff makes a declaration.
    $decl_type = $this->createDeclarationType();
    $donor_user = $this->createUser();
    $staff = $this->createStaffUser($decl_type->id());
    $this->drupalLogin($staff);

    $url = Url::fromRoute('gift_aid_user.context', ['user' => $donor_user->id()]);
    $this->drupalGet($url->toString());
    $this->assertSession()->pageTextContains("Please enter the donor's address");
    $edit = [
      'address[0][address][given_name]' => $this->randomString(),
      'address[0][address][family_name]' => $this->randomString(),
      'address[0][address][address_line1]' => $this->randomString(),
      'address[0][address][locality]' => $this->randomString(),
    ];
    $this->submitForm($edit, 'Next');
    $this->assertSession()->pageTextNotContains('Thank you');
    $this->assertSession()->pageTextContains('Add ' . $decl_type->label());

    // There should now be a stored donor.
    $donor = \Drupal::entityTypeManager()->getStorage('gift_aid_donor')->loadOrCreateByContext($donor_user);
    $this->assertFalse($donor->isNew());

    // Add declaration.
    $edit = [
      'validity' => DeclarationInterface::DECLARATION_INHERENTLY_VALID,
    ];
    $this->assertSession()->fieldNotExists('donor');
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Recorded a ' . $decl_type->label() . ' Gift Aid declaration for ' . $donor_user->getDisplayName() . '.');

    // There should now be a stored declaration.
    $declarations = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration')->loadByContext($donor_user);
    $declaration = reset($declarations);
    $context = $declaration->getDonor()->getContext();
    $this->assertInstanceOf(DeclarationInterface::class, $declaration);
    $this->assertEquals($this->charity->id(), $declaration->getCharity()->id());
    $this->assertEquals('user', $context->getEntityTypeId());
    $this->assertEquals($donor_user->id(), $context->id());
    $this->assertTrue($declaration->isDateBased());
    $this->assertFalse($declaration->hasEndDate());
    $this->assertTrue($declaration->isOngoing());

    // Staff can cancel the declaration.
    $this->clickLink('Cancel');
    $this->assertSession()->pageTextContains('Add Gift Aid cancellation');
    $this->submitForm([], 'Save');
    $this->assertSession()->pageTextContains('Status Not currently valid');

    $declaration = $this->reloadDeclaration($declaration);
    $this->assertTrue($declaration->isCancelled());
    $this->assertEquals($declaration::today(), $declaration->getCancellationDate(TRUE));
    $this->assertEquals(DeclarationInterface::DECLARATION_CANCELLED, $declaration->getStatus());
    $this->assertEquals(DonorInterface::DONOR_DECL_NOT_CURRENT, $donor->getStatus());
    $this->assertFalse($declaration->isOngoing());
  }

  /**
   * Reloads a declaration.
   *
   * @param \Drupal\gift_aid\Entity\DeclarationInterface $declaration
   *   Declaration to reload.
   *
   * @return \Drupal\gift_aid\Entity\DeclarationInterface
   *   The reloaded declaration.
   */
  protected function reloadDeclaration(DeclarationInterface $declaration) {
    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration');
    $storage->resetCache([$declaration->id()]);
    return $storage->load($declaration->id());
  }

}
