<?php

namespace Drupal\Tests\gift_aid\Kernel;

use Drupal\gift_aid\Entity\DeclarationInterface;

/**
 * Tests loading declarations in storage by context.
 *
 * @coversDefaultClass \Drupal\gift_aid\Declaration\DeclarationStorage
 * @group gift_aid
 */
class DeclarationStorageTest extends GiftAidTestBase {

  /**
   * Test the LoadByContext method.
   */
  public function testLoadByContext() {
    $declaration1 = $this->createDeclaration();
    $declaration2 = $this->createDeclaration();
    $another = $this->createDonor(['context' => $this->createUser()]);
    $this->createDeclaration([
      'donor' => $another,
    ]);

    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration');
    $declarations = $storage->loadByContext($this->user);
    $this->assertCount(2, $declarations);
    $this->assertArrayHasKey($declaration1->id(), $declarations);
    $this->assertArrayHasKey($declaration2->id(), $declarations);

    // Specifying charity should make no difference if all declarations use same charity.
    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration');
    $declarations = $storage->loadByContext($this->user, $this->charity);
    $this->assertCount(2, $declarations);

    // Exclude declaration 2 by reassigning to different charity.
    $charity2 = $this->createCharity();
    $declaration2->setCharity($charity2);
    $declarations = $storage->loadByContext($this->user, $this->charity);
    $this->assertCount(2, $declarations);
    $this->assertArrayHasKey($declaration1->id(), $declarations);
  }

  /**
   * Test the hasOngoingByContext method.
   */
  public function testHasOngoingByContext() {
    $declaration1 = $this->createDeclaration();
    $another = $this->createDonor(['context' => $this->createUser()]);
    $declaration2 = $this->createDeclaration([
      'donor' => $another,
    ]);
    $declaration3 = $this->createDeclaration([
      'declared_date' => '2020-01-01',
      'start_date' => '2020-01-01',
      'end_date' => '2021-01-01',
    ]);
    $declaration3->setChangedDate('2020-02-01')->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_PROVISIONAL, $declaration1->getStatus());
    $this->assertEquals(DeclarationInterface::DECLARATION_PROVISIONAL, $declaration2->getStatus());
    $this->assertEquals(DeclarationInterface::DECLARATION_ENDED, $declaration3->getStatus());
    $this->assertTrue($declaration1->isOngoing());
    $this->assertEquals($this->charity, $declaration1->getCharity());
    $this->assertEquals($this->user, $declaration1->getDonor()->getContext());
    $this->assertNull($declaration1->getEndDate());

    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration');
    $this->assertArrayHasKey($declaration1->id(), $storage->loadOngoingByContext($this->user, $this->charity));
    $this->assertArrayHasKey($declaration2->id(), $storage->loadOngoingByContext($another->getContext(), $this->charity));

    // Only active date-based declarations are considered.
    $declaration1->setDateBased(FALSE)->save();
    $this->assertFalse($storage->hasOngoingByContext($this->user, $this->charity));
  }

  /**
   * Test the loadOngoingByContext method.
   */
  public function testLoadOngoingByContext() {
    $storage = \Drupal::entityTypeManager()->getStorage('gift_aid_declaration');

    $another = $this->createDonor(['context' => $this->createUser()]);
    $this->createDeclaration([
      'donor' => $another,
      'declared_date' => '2020-01-01',
    ]);
    $this->assertEmpty($storage->loadOngoingByContext($this->user, $this->charity), "Declarations for the wrong context should be ignored.");

    $this->createDeclaration([
      'declared_date' => '2020-01-01',
      'date_based' => FALSE,
    ]);
    $this->assertEmpty($storage->loadOngoingByContext($this->user, $this->charity), "Declarations that are not date-based are not ongoing.");

    $this->createDeclaration(['declared_date' => '2020-01-01']);
    $this->assertCount(1, $storage->loadOngoingByContext($this->user, $this->charity), "Declaration with no end date is ongoing.");

    $this->createDeclaration(
      [
        'declared_date' => '2020-01-01',
        'end_date' => '2050-01-01',
      ]
    );
    $this->assertCount(2, $storage->loadOngoingByContext($this->user, $this->charity), "Declaration with future end date is ongoing.");

  }

}
