<?php

namespace Drupal\Tests\gift_aid\Kernel;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\gift_aid\Entity\Declaration;
use Drupal\gift_aid\Entity\DeclarationInterface;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\user\UserInterface;

/**
 * Tests the declaration entity.
 *
 * @coversDefaultClass \Drupal\gift_aid\Entity\Declaration
 * @group gift_aid
 */
class DeclarationTest extends GiftAidTestBase {

  // @todo test the revision user and date on a newly created declaration.
  use NodeCreationTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'node',
  ];

  /**
   * Tests CRUD operations.
   *
   * @covers ::getCharityId
   * @covers ::getDeclaredDate
   * @covers ::getStartDate
   * @covers ::getEndDate
   * @covers ::getCancelledDate
   * @covers ::getStatus
   * @covers ::isDateBased
   * @covers ::getContext
   * @covers ::setCharity
   * @covers ::setDeclaredDate
   * @covers ::setStartDate
   * @covers ::setEndDate
   * @covers ::setCancelledDate
   * @covers ::setDeclarationStatus
   * @covers ::setDateBased
   * @covers ::setContext
   */

  /**
   * Test the creation of declaration.
   */
  public function testDefaults() {
    $declaration = $this->createDeclaration();

    $today = Declaration::today();
    $this->assertEquals($this->declarationType->id(), $declaration->getType(), 'Testing declaration type.');
    $this->assertTrue($declaration instanceof Declaration, 'The declaration with date-based is created.');
    $this->assertEquals($this->charity->id(), $declaration->getCharityId(), 'Testing declaration charity.');
    $this->assertEquals($today, $declaration->getDeclaredDate(TRUE), "Declared date should default to current date.");
    $this->assertEquals($today, $declaration->getStartDate(TRUE), "Start date should default to current date.");
    $this->assertNull($declaration->getConfirmationDate(), 'Testing written statement sent date.');
    $this->assertFalse($declaration->isClaimable(), 'Testing claimable.');
    $this->assertTrue($declaration->hasStarted(), 'Testing started.');
    $this->assertFalse($declaration->hasEnded(), 'Testing ended.');
    $this->assertTrue($declaration->isValid(), 'Testing valid.');
    $this->assertTrue($declaration->isOngoing(), 'Testing ongoing.');
    $this->assertEquals(DeclarationInterface::DECLARATION_PROVISIONAL, $declaration->getStatus(), 'Testing declaration status.');
    $this->assertTrue($declaration->isDateBased(), 'Testing date-based.');
    $this->assertEquals($this->user, $declaration->getDonor()->getContext(), 'Testing declarer.');
  }

  /**
   * Test the end date.
   */
  protected function testEndDate() {
    // End date defaults to indefinite.
    $declaration1 = $this->createDeclaration();
    $this->assertNull($declaration1->getEndDate(), 'Testing end date.');
    $this->assertFalse($declaration1->hasEndDate());
    $declaration1->setEndDate('2050-01-01')->save();
    $this->assertTrue($declaration1->hasEndDate(), "Declaration has an end date");
    $this->assertEquals('2050-01-01', $declaration1->getEndDate(TRUE));
    $declaration1->setEndDate(NULL)->save();
    $this->assertFalse($declaration1->hasEndDate());

    // Creating a declaration with specified end date.
    $declaration2 = $this->createDeclaration(
      ['end_date' => '2050-01-01']
    );
    $this->assertTrue($declaration2->hasEndDate(), "Declaration has an end date");
    $this->assertEquals('2050-01-01', $declaration2->getEndDate(TRUE));
  }

  /**
   * Asserts the end date.
   *
   * @param string $expected
   *   The formatted expected end date.
   * @param \Drupal\gift_aid\Entity\DeclarationInterface $declaration
   *   Declaration to check.
   */
  protected function assertEndDate(string $expected, DeclarationInterface $declaration) {
    $declaration = $this->reloadEntity($declaration);
    $actual = $declaration->hasEndDate() ? $declaration->getEndDate(TRUE) : NULL;
    $this->assertSame($expected, $actual);
  }

  /**
   * Test the declared date.
   */
  public function testDeclaredDate() {
    $declaration = $this->createDeclaration();
    $this->assertNotNull($declaration->getDeclaredDate());
    $declaration->setDeclaredDate('2050-01-01')->save();
    $this->assertEquals('2050-01-01', $declaration->getDeclaredDate(TRUE));

    $declaration = $this->createDeclaration(
      ['declared_date' => '2050-01-01']
    );
    $this->assertEquals('2050-01-01', $declaration->getDeclaredDate(TRUE));
  }

  /**
   * Test the creation of a limited declaration, one that is not date-based.
   */
  public function testSelective() {
    $declaration = $this->createDeclaration([
      'date_based' => FALSE,
      'start_date' => "2022-05-20",
    ]);
    $declaration->setChangedDate("2022-05-20")->save();

    $this->assertInstanceOf(Declaration::class, $declaration, 'The declaration should be created.');
    $this->assertFalse($declaration->isDateBased(), "Declaration should not be date-based");
    $this->assertNull($declaration->getStartDate(), 'Start date should be null if declaration is not date-based.');
    $this->assertNull($declaration->getEndDate(), 'Testing end date.');
    $this->assertNull($declaration->getConfirmationDate(), 'Testing written statement sent date.');
    $this->assertEquals(DeclarationInterface::DECLARATION_SELECTIVE, $declaration->getStatus());
  }

  /**
   * Test a start date is required for date-based declarations.
   */
  public function testStartDateRequired() {
    $this->expectException(EntityStorageException::class);
    $this->createDeclaration([
      'date_based' => TRUE,
      'start_date' => NULL,
    ]);
  }

  /**
   * Test the creation of declaration with a node as context.
   */
  public function testNodeContext() {
    $node = $this->createNode();

    $declaration = $this->createDeclaration([
      'context' => $node,
    ]);

    $declaration = Declaration::load($declaration->id());

    $this->assertInstanceOf(Declaration::class, $declaration, 'The declaration should be created.');
    $this->assertEquals($node->id(), $declaration->getDonor()->getContext()->id(), 'Testing node context.');
  }

  /**
   * Test the declaration status.
   */
  public function testStatus() {
    $declaration = $this->createDeclaration();
    $pastDate = new DrupalDateTime('now -30 day');
    $pastDate = $pastDate->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
    $futureDate = new DrupalDateTime('now +30 day');
    $futureDate = $futureDate->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
    $declaration->setChangedDate($pastDate)->save();

    // Declaration starts today by default.
    $this->assertEquals(DeclarationInterface::DECLARATION_ONGOING, $declaration->getStatus(), 'Declaration should be ongoing by default.');

    $declaration->setStartDate($pastDate)->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_PROVISIONAL, $declaration->getStatus(), 'Declaration should be provisional.');

    $declaration->setChangedDate($pastDate)->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_ONGOING, $declaration->getStatus(), 'Declaration should be ongoing if start date is in the past.');

    $declaration->setStartDate($futureDate)->save();
    $declaration->setChangedDate($pastDate)->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_PENDING, $declaration->getStatus(), 'Declaration should be pending if start date is in the future.');

    $declaration->setStartDate($pastDate)->setEndDate($pastDate)->save();
    $declaration->setChangedDate($pastDate)->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_ENDED, $declaration->getStatus(), 'Declaration should be ended if end date is in the past.');

    $declaration->setDateBased(FALSE)->save();
    $declaration->setChangedDate($pastDate)->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_SELECTIVE, $declaration->getStatus(), 'Declaration should be selective if not date-based.');

    $declaration->setValidity(DeclarationInterface::DECLARATION_VALID_IF_CONFIRMED)->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_CONFIRMATION_MISSING, $declaration->getStatus());
    $this->assertFalse($declaration->isConfirmationSent());
    $declaration->setConfirmationDate($pastDate)->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_PROVISIONAL, $declaration->getStatus());
    $this->assertTrue($declaration->isConfirmationSent());

    $declaration->setValidity(DeclarationInterface::DECLARATION_INVALID);
    $declaration->save();
    $this->assertEquals(DeclarationInterface::DECLARATION_INVALID, $declaration->getStatus(), 'Declaration should be invalid.');
  }

  /**
   * Tests declaration entity and field access.
   */
  public function testAccess() {
    // Fields only set by admin if locked.
    $critical_fields = ['date_based', 'start_date', 'end_date'];

    // Fields only set by admin.
    $admin_fields = ['confirmation_date', 'changed_date', 'cancellation_date'];

    $admin = $this->createUser(['administer gift aid declarations'])->setUsername('Admin');
    $staff = $this->createStaffUser($this->declarationType->id())->setUsername('Staff');
    $donor = $this->createUser(['cancel gift aid declaration', 'make gift aid declaration'])->setUsername('Donor');

    $gift_aid_donor = $this->createDonor()->setContext($donor);
    $declaration = Declaration::create(['type' => $this->declarationType->id()])
      ->setDonor($gift_aid_donor)
      ->setEvidence($this->evidence)
      ->setExplanation('I am a UK tax payer')
      ->setValidity(DeclarationInterface::DECLARATION_INHERENTLY_VALID);
    $declaration->save();
    $this->assertFalse($declaration->isClaimable());

    // Staff and admins can create, update and view declarations.
    $this->assertTrue($declaration->access('create', $admin));
    $this->assertTrue($declaration->access('create', $staff));
    $this->assertFalse($declaration->access('create', $donor));
    $this->assertTrue($declaration->access('update', $admin));
    $this->assertTrue($declaration->access('update', $staff));
    $this->assertFalse($declaration->access('update', $donor));
    $this->assertTrue($declaration->access('view', $admin));
    $this->assertTrue($declaration->access('view', $staff));
    $this->assertFalse($declaration->access('view', $donor));

    // Only admins can delete declarations.
    $this->assertTrue($declaration->access('delete', $admin));
    $this->assertFalse($declaration->access('delete', $staff));
    $this->assertFalse($declaration->access('delete', $donor));

    // Staff can still edit critical fields on a recently created declaration.
    $this->assertFieldAccess($critical_fields, $staff, TRUE, $declaration);
    $this->assertFieldAccess($admin_fields, $staff, FALSE, $declaration);
    $this->assertFieldAccess($critical_fields, $admin, TRUE, $declaration);
    $this->assertFieldAccess($admin_fields, $admin, TRUE, $declaration);

    // Staff cannot edit critical fields on a locked declaration, but admins can.
    $declaration->setChangedDate(new DrupalDateTime('2 months ago'))->save();
    \Drupal::entityTypeManager()->getAccessControlHandler('gift_aid_declaration')->resetCache();
    $this->assertTrue($declaration->isLocked());
    $this->assertTrue($declaration->isClaimable());
    $this->assertTrue($declaration->access('update', $admin));
    $this->assertTrue($declaration->access('update', $staff));
    $this->assertFieldAccess($critical_fields, $staff, FALSE, $declaration);
    $this->assertFieldAccess($admin_fields, $staff, FALSE, $declaration);
    $this->assertFieldAccess($critical_fields, $admin, TRUE, $declaration);
    $this->assertFieldAccess($admin_fields, $admin, TRUE, $declaration);
  }

  /**
   * Checks field access.
   *
   * @param string[] $fields
   *   Array of field names to check.
   * @param \Drupal\user\UserInterface $user
   *   User to check access for.
   * @param bool $access
   *   Expected access.
   * @param \Drupal\gift_aid\Entity\DeclarationInterface $declaration
   *   Declaration to check.
   */
  protected function assertFieldAccess(array $fields, UserInterface $user, bool $access, DeclarationInterface $declaration) {
    $access_text = $access ? 'true' : 'false';
    $user_name = $user->getDisplayName();
    foreach ($fields as $field) {
      $this->assertEquals($access, $declaration->get($field)->access('edit', $user), "Access should be $access_text for field $field and user $user_name");
    }
  }

}
