<?php

namespace Drupal\Tests\gift_aid\Kernel;

use Drupal\gift_aid\Entity\Donor;
use Drupal\Tests\node\Traits\NodeCreationTrait;

/**
 * Tests the donor entity.
 *
 * @coversDefaultClass \Drupal\gift_aid\Entity\Donor
 * @group gift_aid
 */
class DonorTest extends GiftAidTestBase {

  use NodeCreationTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'node',
  ];

  /**
   * Test the creation of donor.
   *
   * @covers ::getAddress
   * @covers ::setAddress
   * @covers ::getAddressName
   */
  public function testDefaults() {
    $donor = $this->createDonor();
    $this->assertTrue($donor instanceof Donor);
    $this->assertEquals($this->user, $donor->getContext());
    $address = $donor->getAddress();
    $this->assertEquals($address->getCountryCode(), 'GB');
    $this->assertEquals($address->getAddressLine1(), 'My house');
    $this->assertEquals($address->getGivenName(), 'John');
    $this->assertEquals($donor->getAddressName(), 'John Smith');
  }

  /**
   * Test the creation of donor with a node as context.
   *
   * @covers ::getContext
   * @covers ::setContext
   */
  public function testNodeContext() {
    $node = $this->createNode();

    $donor = $this->createDonor([
      'context' => $node,
    ]);

    $donor = Donor::load($donor->id());

    $this->assertInstanceOf(Donor::class, $donor, 'The donor should be created.');
    $this->assertEquals($node->id(), $donor->getContext()->id(), 'Testing node context.');
  }

  /**
   * Tests Donor access control.
   */
  public function testAccess() {
    $admin = $this->createUser(['administer gift aid declarations'])->setUsername('Admin');
    $staff = $this->createUser(['cancel gift aid declarations made by others', 'record gift aid declarations made by others'])->setUsername('Staff');
    $donor = $this->createUser(['cancel gift aid declaration', 'make gift aid declaration'])->setUsername('Donor');
    $nobody = $this->createUser()->setUsername('Nobody');

    // When creating a declaration, the only restriction is that
    // donors cannot access staff fields.
    $dd = $this->createDonor(save: FALSE);
    $this->assertTrue($dd->access('create', $admin));
    $this->assertTrue($dd->access('create', $staff));
    $this->assertFalse($dd->access('create', $donor));
    $this->assertFalse($dd->access('create', $nobody));

    // Staff can edit donor.
    $dd->save();
    $this->assertTrue($dd->access('update', $admin));
    $this->assertTrue($dd->access('update', $staff));
    $this->assertFalse($dd->access('update', $donor));
    $this->assertFalse($dd->access('update', $nobody));

    // @todo Block deleting if there are declarations.
    $this->assertTrue($dd->access('delete', $admin));
    $this->assertFalse($dd->access('delete', $staff));
    $this->assertFalse($dd->access('delete', $donor));
    $this->assertFalse($dd->access('delete', $nobody));
  }

}
