<?php

namespace Drupal\Tests\gift_aid\Traits;

use Drupal\gift_aid\Entity\Charity;
use Drupal\gift_aid\Entity\Donor;
use Drupal\gift_aid\Entity\Declaration;
use Drupal\gift_aid\Entity\DeclarationType;
use Drupal\gift_aid\Entity\DeclarationInterface;
use Drupal\gift_aid\Entity\Evidence;
use Drupal\gift_aid\Entity\EvidenceType;

/**
 * Helper methods for gift aid tests.
 */
trait GiftAidTestTrait {

  /**
   * A sample user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user;

  /**
   * A sample donor.
   *
   * @var \Drupal\gift_aid\Entity\DonorInterface
   */
  protected $donor;

  /**
   * A sample charity.
   *
   * @var \Drupal\gift_aid\Entity\CharityInterface
   */
  protected $charity;

  /**
   * A sample declaration type.
   *
   * @var \Drupal\gift_aid\Entity\DeclarationTypeInterface
   */
  protected $declarationType;

  /**
   * A sample evidence type.
   *
   * @var \Drupal\gift_aid\Entity\EvidenceTypeInterface
   */
  protected $evidenceType;

  /**
   * Sample evidence.
   *
   * @var \Drupal\gift_aid\Entity\EvidenceInterface
   */
  protected $evidence;

  /**
   * Create default entities.
   */
  protected function createGiftAidDefaultEntities() {
    $user = $this->createUser(['administer gift aid declarations'])->setEmail('declarationtest@example.com');
    $user_storage = \Drupal::entityTypeManager()->getStorage('user');
    $user_storage->resetCache([$user->id()]);
    $this->user = $user_storage->load($user->id());
    $this->charity = $this->createCharity(['is_default' => TRUE]);
    $this->evidenceType = $this->createEvidenceType();
    $this->evidence = $this->createEvidence();
    $this->donor = $this->createDonor();
    $this->declarationType = $this->createDeclarationType();
  }

  /**
   * Create a charity.
   *
   * @return \Drupal\gift_aid\Entity\CharityInterface
   *   A new Charity entity.
   */
  public function createCharity($values = []) {
    $name = $this->randomMachineName();
    $defaults = [
      'name' => 'Charity name ' . $name,
      'registered_name' => 'Charity registered name ' . $name,
      'registration_number' => 'Charity registration number ' . $name,
      'registered_address' => 'Charity registered address ' . $name,
    ];
    $charity = Charity::create($values + $defaults);
    $charity->save();
    return Charity::load($charity->id());
  }

  /**
   * Create a donor.
   *
   * @return \Drupal\gift_aid\Entity\DonorInterface
   *   A donor entity.
   */
  protected function createDonor($values = [], $save = TRUE) {
    $defaults = [
      'context' => [
        'target_id' => $this->user->id(),
        'target_type' => 'user',
      ],
      'address' => [
        'country_code' => 'GB',
        'locality' => 'Birmingham',
        'postal_code' => 'B11 1AA',
        'address_line1' => 'My house',
        'given_name' => 'John',
        'family_name' => 'Smith',
      ],
    ];
    $values = $values + $defaults;
    $donor = Donor::create($values);
    if ($save) {
      $donor->save();
    }
    return $donor;
  }

  /**
   * Create an evidence type.
   *
   * @return \Drupal\gift_aid\Entity\EvidenceTypeInterface
   *   A new EvidenceType entity.
   */
  public function createEvidenceType() {
    $name = $this->randomMachineName();
    $evidence_type = EvidenceType::create([
      'id' => 'evidence_type_' . $name,
      'label' => 'Evidence type ' . $name,
      'written_record_required' => FALSE,
    ]);
    $evidence_type->save();
    return EvidenceType::load($evidence_type->id());
  }

  /**
   * Create evidence.
   *
   * @return \Drupal\gift_aid\Entity\EvidenceInterface
   *   A new Evidence entity.
   */
  public function createEvidence() {
    $evidence = Evidence::create([
      'type' => $this->evidenceType->id(),
    ]);
    $evidence->save();
    return Evidence::load($evidence->id());
  }

  /**
   * Create a declaration type.
   *
   * @return \Drupal\gift_aid\Entity\DeclarationTypeInterface
   *   A new DeclarationType entity.
   */
  public function createDeclarationType() {
    $name = $this->randomMachineName();
    $declaration_type = DeclarationType::create([
      'id' => 'declaration_type_' . $name,
      'label' => 'Declaration type ' . $name,
      'written_record_required' => FALSE,
    ]);
    $declaration_type->save();
    return DeclarationType::load($declaration_type->id());
  }

  /**
   * Creates a 'staff' user with permissions to manage gift aid for others.
   *
   * @param string $bundle
   *   Declaration bundle type to give access to.
   *
   * @return \Drupal\user\Entity\User|false
   *   A fully loaded user object with pass_raw property, or FALSE if account
   *   creation fails.
   */
  protected function createStaffUser(string $bundle = 'recording') {
    $permissions = [
      'cancel gift aid declarations made by others',
      'record gift aid declarations made by others',
      'view gift aid declarations',
      "create $bundle declarations",
    ];
    return $this->createUser($permissions);
  }

  /**
   * Create a declaration.
   *
   * @param array $values
   *   Values to set on the declaration.
   *
   * @return \Drupal\gift_aid\Entity\DeclarationInterface
   *   A declaration entity.
   */
  protected function createDeclaration(array $values = []) {
    $defaults = [
      'type' => $this->declarationType->id(),
      'charity' => $this->charity->id(),
      'donor' => $this->donor->id(),
      'evidence' => $this->evidence->id(),
      'explanation' => 'I am a UK taxpayer and understand my liabilities.',
      'validity' => DeclarationInterface::DECLARATION_INHERENTLY_VALID,
    ];
    $values = $values + $defaults;
    $declaration = Declaration::create($values);
    $declaration->save();
    return $declaration;
  }

}
