<?php

namespace Drupal\Tests\ginvite\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\ginvite\Plugin\GroupContentEnabler\GroupInvitation;
use Drupal\group\Entity\GroupContentInterface;
use Drupal\Tests\group\Functional\GroupBrowserTestBase;
use Drupal\user\Entity\User;

/**
 * Tests the behavior of the group invite functionality.
 *
 * @group group
 */
class GroupInviteTest extends GroupBrowserTestBase {

  use StringTranslationTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The invitation manager.
   *
   * @var \Drupal\ginvite\GroupInvitationManager
   */
  protected $groupInvitationManager;

  /**
   * The group we will use to test methods on.
   *
   * @var \Drupal\group\Entity\Group
   */
  protected $group;

  /**
   * The normal user we will use.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $account;

  /**
   * The group content type for group membership request.
   *
   * @var \Drupal\group\Entity\GroupContentTypeInterface
   */
  protected $groupContentType;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'group',
    'group_test_config',
    'ginvite',
  ];

  /**
   * Gets the global (site) permissions for the group creator.
   *
   * @return string[]
   *   The permissions.
   */
  protected function getGlobalPermissions() {
    return [
      'view the administration theme',
      'access administration pages',
      'access group overview',
      'create default group',
      'create other group',
      'administer group',
      'bypass group access',
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->group = $this->createGroup(['uid' => $this->groupCreator->id()]);

    $this->account = $this->drupalCreateUser();
    $this->group->addMember($this->account);
    $this->group->save();

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->groupInvitationManager = $this->container->get('ginvite.group_invitation_manager');

    $this->installPlugin();
  }

  /**
   * Check routes access.
   *
   * @param \Drupal\group\Entity\GroupContentInterface $group_invitation
   *   Group invitation.
   * @param int $status
   *   HTTP status.
   */
  private function checkRoutesAccess(GroupContentInterface $group_invitation, $status) {
    $this->drupalGet("/ginvite/{$group_invitation->id()}/accept");
    $this->assertSession()->statusCodeEquals($status);

    $this->drupalGet("/ginvite/{$group_invitation->id()}/decline");
    $this->assertSession()->statusCodeEquals($status);
  }

  /**
   * Install group invitation plugin.
   */
  private function installPlugin() {
    $this->drupalLogin($this->groupCreator);

    // Install and configure the Group Invitation plugin.
    $this->drupalGet('/admin/group/content/install/default/group_invitation');
    $this->submitForm([], 'Install plugin');
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalGet('/admin/group/content/manage/default-group_invitation');
    $this->submitForm(['invitation_bypass_form' => 1], 'Save configuration');
    $this->assertSession()->statusCodeEquals(200);

    $this->groupContentType = $this->entityTypeManager->getStorage('group_content_type')->load('default-group_invitation');

    // Allow to see invitations.
    $role = $this->group->getGroupType()->getOutsiderRole();
    $role->grantPermissions(['view group invitations']);
    $role->save();

    drupal_flush_all_caches();
  }

  /**
   * Create invites and test general group invite behavior.
   */
  public function testInviteRolePermission() {
    $this->drupalLogin($this->account);

    $role = $this->group->getGroupType()->getMemberRole();
    $role->grantPermissions(['invite users to group']);
    $role->save();

    // Verify the user cannot add roles to users on invite.
    $this->drupalGet("/group/{$this->group->id()}/content/add/group_invitation");
    $this->assertSession()->fieldNotExists('group_roles[default-custom]');

    // Add permissions to administer members to members of the group.
    $role = $this->group->getGroupType()->getMemberRole();
    $role->grantPermissions(['administer members']);
    $role->save();

    // Verify the normal member without the permission cannot add roles
    // to users on invite.
    $this->drupalGet("/group/{$this->group->id()}/content/add/group_invitation");
    $this->assertSession()->fieldExists('group_roles[default-custom]');
  }

  /**
   * We want to be sure user gets the group role.
   */
  public function testRoleAssigment() {
    $account = $this->drupalCreateUser();

    $this->drupalLogin($account);

    $role_name = 'default-custom';
    $group_invitation = $this->groupInvitationManager->createInvitation($this->group, $account->getEmail(), $account->id(), [$role_name]);
    $group_invitation->save();

    // Install and configure the Group Invitation plugin.
    $this->drupalGet("/ginvite/{$group_invitation->id()}/accept");
    $this->assertSession()->statusCodeEquals(200);

    $group_membership = $this->group->getMember($account);
    $this->assertTrue(in_array($role_name, array_keys($group_membership->getRoles())), 'Role has been found');
  }

  /**
   * Owner can access own invitations.
   */
  public function testAccessOwnInvitation() {
    // Allow to see invitations.
    $role = $this->group->getGroupType()->getOutsiderRole();
    $role->grantPermissions(['view group invitations']);
    $role->save();

    $account = $this->drupalCreateUser();
    $this->drupalLogin($account);

    $group_invitation = $this->groupInvitationManager->createInvitation($this->group, $account->getEmail(), $account->id());
    $group_invitation->save();

    $this->drupalGet("/ginvite/{$group_invitation->id()}/accept");
    $this->assertSession()->statusCodeEquals(200);

    $account = $this->drupalCreateUser();
    $this->drupalLogin($account);

    $group_invitation = $this->groupInvitationManager->createInvitation($this->group, $account->getEmail(), $account->id());
    $group_invitation->save();

    $this->drupalGet("/ginvite/{$group_invitation->id()}/decline");
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Not owner can't access invitations.
   */
  public function testNotOwnerAccessRoutes() {
    $account = $this->drupalCreateUser();

    $group_invitation = $this->groupInvitationManager->createInvitation($this->group, $account->getEmail(), $account->id());
    $group_invitation->save();

    $not_owner_user = $this->drupalCreateUser();
    $this->drupalLogin($not_owner_user);

    // As not owner of invitation I can't accept or decline it.
    $this->checkRoutesAccess($group_invitation, 403);
  }

  /**
   * Access not pending invitation.
   */
  public function testAccessNotPendingInvitation() {
    $account = $this->drupalCreateUser();

    $this->drupalLogin($account);

    // Allow to see invitations.
    $role = $this->group->getGroupType()->getOutsiderRole();
    $role->grantPermissions(['view group invitations']);
    $role->save();
  }

  /**
   * Check bulk operations routes access.
   */
  public function testBulkInvitationRoutes() {
    $account = $this->drupalCreateUser();

    $this->drupalLogin($account);

    // Allow to invite user without in bulk.
    $role = $this->group->getGroupType()->getOutsiderRole();
    $role->grantPermissions(['invite users to group']);
    $role->save();

    $this->drupalGet("/group/{$this->group->id()}/invite-members");
    $this->assertSession()->statusCodeEquals(403);
    $this->drupalGet("/group/{$this->group->id()}/invite-members/confirm");
    $this->assertSession()->statusCodeEquals(403);

    // Allow to invite users in bulk.
    $role->grantPermissions(['bulk invite users to group']);
    $role->save();

    $this->drupalGet("/group/{$this->group->id()}/invite-members");
    $this->assertSession()->statusCodeEquals(200);
    $this->drupalGet("/group/{$this->group->id()}/invite-members/confirm");
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Unblock user during registration, if unblock_invitees option enabled.
   *
   * @param int $unblock_invitees
   *   Option unblock_invitees
   * @param bool $user_status
   *   Current user status.
   * @param $expected_user_status
   *   Expected user status.
   *
   * @dataProvider groupInvitationUserRegistrationData
   */

  public function testUnlbockRegisteredUser($unblock_invitees, $user_status, $expected_user_status) {
    $email = $this->randomMachineName() . '@domain.com';

    // Enable unblock_invitees option.
    $this->groupContentType->updateContentPlugin(['unblock_invitees' => $unblock_invitees]);

    $group_invitation = $this->groupInvitationManager->createInvitation($this->group, $email);
    $group_invitation->save();

    $account = $this->drupalCreateUser([], NULL, FALSE, [
      'mail' => $email,
      'status' => $user_status,
    ]);

    // Reload account.
    $user = User::load($account->id());

    if ($expected_user_status) {
      $this->assertTrue($user->isActive());
    }
    else {
      $this->assertFalse($user->isActive());
    }
  }

  /**
   * Data provider for testUnlbockRegisteredUser().
   *
   * @return array
   *   Data to check unlock functionality.
   */
  public function groupInvitationUserRegistrationData() {
    return [
      // Each array contains [email, unblock_invitees option, user status, expected user status].
      [1, FALSE, TRUE],
      [1, TRUE, TRUE],
      [0, FALSE, FALSE],
      [0, TRUE, TRUE],
    ];
  }

}
