<?php

namespace Drupal\ginvite;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group\Entity\GroupContent;
use Drupal\group\Entity\GroupContentInterface;
use Drupal\group\Entity\GroupInterface;

/**
 * Group Invitation Manager class.
 */
class GroupInvitationManager {

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Group content type.
   *
   * @var \Drupal\group\Entity\Storage\GroupContentTypeStorageInterface
   */
  protected $groupContentTypeStorage;

  /**
   * Group Invitation Manager constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->groupContentTypeStorage = $this->entityTypeManager->getStorage('group_content_type');
  }

  /**
   * Create group invitation group content.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   Group.
   * @param string|null $email
   *   Email.
   * @param int|null $user_id
   *   User id.
   * @param array|null $group_roles
   *   Group roles.
   *
   * @return \Drupal\group\Entity\GroupContentInterface
   *   Group invitation group content.
   */
  public function createInvitation(GroupInterface $group, $email = NULL, $user_id = 0, array $group_roles = []) {
    $group_type = $group->getGroupType();
    $plugin_id = 'group_invitation';
    if (!$group_type->hasContentPlugin('group_invitation')) {
      throw new \Exception("Group invitation is not install for group type {$group_type->label()}");
    }

    $group_content_roles = [];
    if (!empty($group_roles)) {
      $group_type_roles = array_keys($group_type->getRoles());
      foreach ($group_roles as $group_role) {
        if (!in_array($group_role, $group_type_roles)) {
          throw new \Exception("Group role $group_role does not exist for group type {$group_type->label()}");
        }

        $group_content_roles[] = ['target_id' => $group_role];
      }
    }

    $content_type_config_id = $group_type
      ->getContentPlugin($plugin_id)
      ->getContentTypeConfigId();

    $group_content = GroupContent::create([
      'type' => $content_type_config_id,
      'gid' => $group->id(),
      'entity_id' => $user_id,
      'invitee_mail' => $email,
      'group_roles' => $group_content_roles,
    ]);

    return $group_content;
  }

  /**
   * Create group membership based on group invitation.
   *
   * @param \Drupal\group\Entity\GroupContentInterface $group_invitation_content
   *   Group.
   *
   * @return \Drupal\group\Entity\GroupContentInterface
   *   Group membership content.
   */
  public function createMember(GroupContentInterface $group_invitation_content) {
    $group = $group_invitation_content->getGroup();
    $group_type = $group->getGroupType();
    $group_type_id = $group_type->id();

    $user = $group_invitation_content->get('entity_id')->entity;
    $group_membership = $group->getMember($user);
    // User already a member.
    if (!empty($group_membership)) {
      return $group_membership->getGroupContent();
    }

    $plugin_id = 'group_membership';

    $content_type_config_id = $group_type
      ->getContentPlugin($plugin_id)
      ->getContentTypeConfigId();

    $group_membership = GroupContent::create([
      'type' => $content_type_config_id,
      'entity_id' => $user->id(),
      'plugin_id' => $plugin_id,
      'group_type' => $group_type_id,
      'gid' => $group->id(),
      'uid' => $group_invitation_content->getOwnerId(),
      'group_roles' => $group_invitation_content->get('group_roles')->getValue(),
    ]);

    return $group_membership;
  }

}
